# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (not (need-ready)) is a missing predicate in preconditions.
        # We need to ensure that before executing any skill, (not (need-ready)) holds.
        # If (need-ready) is True, we must execute the 'execute_go_ready' skill to clear it.

        # We'll assume the task object provides a method to check predicates in the current state.
        # If not, we attempt to infer from observation or skip if not available.

        def is_need_ready():
            # Try to check if 'need-ready' is True in the current state
            # This is a placeholder; actual implementation may depend on the task API
            try:
                if hasattr(task, 'check_predicate'):
                    return task.check_predicate('need-ready')
                # Fallback: try to infer from obs if possible
                if isinstance(obs, dict) and 'need-ready' in obs:
                    return obs['need-ready']
            except Exception:
                pass
            # If unknown, assume False (safe default)
            return False

        # Helper: Ensure (not (need-ready)) holds before any skill
        def ensure_not_need_ready(current_location):
            if is_need_ready():
                print("[Exploration] (need-ready) is True, executing execute_go_ready to clear it.")
                try:
                    obs2, reward2, done2 = execute_go_ready(env, task, from_location=current_location)
                    return obs2, reward2, done2
                except Exception as e:
                    print("[Exploration] Error executing execute_go_ready:", e)
            return obs, 0, False

        # === Example Plan Execution ===
        # For demonstration, let's assume a simple plan:
        # 1. Move to a location with an object on the floor
        # 2. Pick the object
        # 3. Place the object into a drawer (if available)
        # 4. Push the drawer closed
        # (You should replace this with the actual oracle plan for your task.)

        # For this example, we will:
        # - Find an object on the floor and its location
        # - Find a drawer and its location
        # - Execute the sequence: go -> pick -> go -> place -> push

        # Find an object on the floor
        object_on_floor = None
        object_location = None
        for obj_name, info in positions.items():
            if info.get('on_floor', False):
                object_on_floor = obj_name
                object_location = info.get('location')
                break

        # Find a drawer and its location
        drawer_name = None
        drawer_location = None
        for obj_name, info in positions.items():
            if info.get('type', '') == 'drawer':
                drawer_name = obj_name
                drawer_location = info.get('location')
                break

        # Assume robot starts at 'ready-pose'
        current_location = 'ready-pose'
        done = False

        # 1. Move to object location
        if object_on_floor and object_location:
            obs, reward, done = ensure_not_need_ready(current_location)
            if done:
                print("[Task] Task ended during ensure_not_need_ready before move.")
                return
            try:
                print(f"[Task] Moving from {current_location} to {object_location} to approach {object_on_floor}")
                obs, reward, done = execute_go(env, task, from_location=current_location, to_location=object_location)
                current_location = object_location
                if done:
                    print("[Task] Task ended after move.")
                    return
            except Exception as e:
                print("[Task] Error during execute_go:", e)
                return

            # 2. Pick the object
            obs, reward, done = ensure_not_need_ready(current_location)
            if done:
                print("[Task] Task ended during ensure_not_need_ready before pick.")
                return
            try:
                print(f"[Task] Picking up {object_on_floor} at {object_location}")
                obs, reward, done = execute_pick(env, task, obj_name=object_on_floor, location=object_location)
                if done:
                    print("[Task] Task ended after pick.")
                    return
            except Exception as e:
                print("[Task] Error during execute_pick:", e)
                return

            # 3. Move to drawer location
            if drawer_name and drawer_location:
                obs, reward, done = ensure_not_need_ready(current_location)
                if done:
                    print("[Task] Task ended during ensure_not_need_ready before move to drawer.")
                    return
                try:
                    print(f"[Task] Moving from {current_location} to {drawer_location} to approach {drawer_name}")
                    obs, reward, done = execute_go(env, task, from_location=current_location, to_location=drawer_location)
                    current_location = drawer_location
                    if done:
                        print("[Task] Task ended after move to drawer.")
                        return
                except Exception as e:
                    print("[Task] Error during execute_go to drawer:", e)
                    return

                # 4. Place the object into the drawer
                obs, reward, done = ensure_not_need_ready(current_location)
                if done:
                    print("[Task] Task ended during ensure_not_need_ready before place.")
                    return
                try:
                    print(f"[Task] Placing {object_on_floor} into {drawer_name} at {drawer_location}")
                    obs, reward, done = execute_place(env, task, obj_name=object_on_floor, drawer_name=drawer_name, location=drawer_location)
                    if done:
                        print("[Task] Task ended after place.")
                        return
                except Exception as e:
                    print("[Task] Error during execute_place:", e)
                    return

                # 5. Push the drawer closed
                obs, reward, done = ensure_not_need_ready(current_location)
                if done:
                    print("[Task] Task ended during ensure_not_need_ready before push.")
                    return
                try:
                    print(f"[Task] Pushing {drawer_name} closed at {drawer_location}")
                    obs, reward, done = execute_push(env, task, drawer_name=drawer_name, location=drawer_location)
                    if done:
                        print("[Task] Task ended after push.")
                        return
                except Exception as e:
                    print("[Task] Error during execute_push:", e)
                    return

        else:
            print("[Task] No object on floor or drawer found in positions. Exploration phase only.")

        # === End of Plan ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
