# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # In the domain, many actions set (need-ready) as an effect, and require (not (need-ready)) as a precondition.
        # Therefore, after any action that sets (need-ready), we must execute the "go ready" skill to clear it.

        # For demonstration, let's assume a generic plan:
        # 1. Pick up an object from the floor (sets need-ready)
        # 2. Go to ready pose (clears need-ready)
        # 3. Place the object into a drawer (sets need-ready)
        # 4. Go to ready pose (clears need-ready)
        # 5. Push the drawer closed (does not set need-ready)
        # (The actual plan may differ, but this structure ensures we handle need-ready correctly.)

        # For this example, we will use placeholder object and location names.
        # Replace 'object_1', 'drawer_1', 'location_1', etc. with actual names from your environment.

        # Example object and location names (replace with actual ones)
        object_name = None
        drawer_name = None
        handle_name = None
        location_floor = None
        location_drawer = None
        ready_pose = 'ready-pose'

        # Try to infer object, drawer, and location names from positions dict
        for key in positions:
            if 'drawer' in key and drawer_name is None:
                drawer_name = key
            elif 'handle' in key and handle_name is None:
                handle_name = key
            elif object_name is None and ('object' in key or 'item' in key or 'ball' in key):
                object_name = key
            elif location_floor is None and ('floor' in key or 'table' in key or 'loc' in key):
                location_floor = key

        # Fallbacks if not found
        if object_name is None:
            object_name = 'object_1'
        if drawer_name is None:
            drawer_name = 'drawer_1'
        if location_floor is None:
            location_floor = 'location_1'
        if location_drawer is None:
            location_drawer = 'location_2'

        # For demonstration, we assume the robot starts at location_floor
        current_location = location_floor

        # === Step 1: Pick up object from floor ===
        print(f"[Task] Picking up {object_name} from {location_floor}")
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                obj=object_name,
                location=location_floor
            )
        except Exception as e:
            print(f"[Error] Failed to pick {object_name}: {e}")
            return

        # After pick, (need-ready) is set. Must go to ready pose.
        print("[Task] Going to ready pose to clear (need-ready)")
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=current_location,
                to_location=ready_pose
            )
            current_location = ready_pose
        except Exception as e:
            print(f"[Error] Failed to go to ready pose: {e}")
            return

        # === Step 2: Move to drawer location ===
        print(f"[Task] Moving to drawer location: {location_drawer}")
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=current_location,
                to_location=location_drawer
            )
            current_location = location_drawer
        except Exception as e:
            print(f"[Error] Failed to move to drawer location: {e}")
            return

        # === Step 3: Place object into drawer ===
        print(f"[Task] Placing {object_name} into {drawer_name}")
        try:
            obs, reward, done = execute_place(
                env,
                task,
                obj=object_name,
                drawer=drawer_name,
                location=location_drawer
            )
        except Exception as e:
            print(f"[Error] Failed to place {object_name} into {drawer_name}: {e}")
            return

        # After place, (need-ready) is set. Must go to ready pose.
        print("[Task] Going to ready pose to clear (need-ready) after placing")
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=current_location,
                to_location=ready_pose
            )
            current_location = ready_pose
        except Exception as e:
            print(f"[Error] Failed to go to ready pose after placing: {e}")
            return

        # === Step 4: Move to drawer location again to push drawer closed ===
        print(f"[Task] Moving to drawer location to push drawer closed: {location_drawer}")
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=current_location,
                to_location=location_drawer
            )
            current_location = location_drawer
        except Exception as e:
            print(f"[Error] Failed to move to drawer location for push: {e}")
            return

        # === Step 5: Push drawer closed ===
        print(f"[Task] Pushing {drawer_name} closed")
        try:
            obs, reward, done = execute_push(
                env,
                task,
                drawer=drawer_name,
                location=location_drawer
            )
        except Exception as e:
            print(f"[Error] Failed to push {drawer_name} closed: {e}")
            return

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()

# Notes:
# - This code demonstrates the correct handling of the (need-ready) predicate as required by the domain and feedback.
# - The exploration phase is implicitly handled by ensuring that after any action that sets (need-ready), the robot executes the "go ready" skill (execute_go to ready-pose).
# - All skill invocations use only the predefined skills from skill_code.
# - Replace placeholder object/drawer/location names with actual names from your environment as needed.
# - Error handling is included for robustness.