# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # In the domain, many actions set (need-ready) as an effect, and require (not (need-ready)) as a precondition.
        # The only action that clears (need-ready) is execute_go_ready.
        # Therefore, after any action that sets (need-ready), we must call execute_go_ready to clear it.

        # === Example Plan Execution ===
        # For demonstration, we will:
        # 1. Pick up an object from the floor (execute_pick_object)
        # 2. Go to ready pose (execute_go_ready) to clear (need-ready)
        # 3. Place the object into a drawer (execute_place_to_drawer)
        # 4. Go to ready pose (execute_go_ready) again

        # For this example, we assume the following:
        # - There is at least one object on the floor (not a handle)
        # - There is at least one drawer, and its state is open and not full
        # - The robot starts at a location where it can pick the object

        # You may need to adapt the object and drawer names to your environment

        # Find a pickable object on the floor (not a handle)
        pick_obj = None
        pick_obj_pos = None
        for obj_name, obj_info in positions.items():
            if obj_info.get('on_floor', False) and not obj_info.get('is_handle', False):
                pick_obj = obj_name
                pick_obj_pos = obj_info.get('position')
                pick_obj_location = obj_info.get('location')
                break

        if pick_obj is None:
            print("[Exploration] No suitable object found on the floor to pick.")
            return

        # Find a suitable drawer
        drawer_name = None
        drawer_location = None
        for obj_name, obj_info in positions.items():
            if obj_info.get('type', '') == 'drawer':
                if obj_info.get('open', False) and not obj_info.get('full', False):
                    drawer_name = obj_name
                    drawer_location = obj_info.get('location')
                    break

        if drawer_name is None:
            print("[Exploration] No suitable open drawer found for placing.")
            return

        # Get robot's current location
        robot_location = None
        for obj_name, obj_info in positions.items():
            if obj_name == 'robot':
                robot_location = obj_info.get('location')
                break

        if robot_location is None:
            # Fallback: try to get from observation
            robot_location = 'ready-pose'

        # === Step 1: Move to pick location if needed ===
        if robot_location != pick_obj_location:
            try:
                print(f"[Task] Moving robot from {robot_location} to {pick_obj_location} to pick {pick_obj}")
                obs, reward, done = execute_go(env, task, robot_location, pick_obj_location)
                robot_location = pick_obj_location
            except Exception as e:
                print(f"[Error] Failed to move to pick location: {e}")
                return

        # === Step 2: Pick the object from the floor ===
        try:
            print(f"[Task] Picking up object {pick_obj} at {pick_obj_location}")
            obs, reward, done = execute_pick_object(env, task, pick_obj, pick_obj_location)
        except Exception as e:
            print(f"[Error] Failed to pick object: {e}")
            return

        # === Step 3: Go to ready pose to clear (need-ready) ===
        try:
            print(f"[Task] Going to ready pose to clear (need-ready)")
            obs, reward, done = execute_go_ready(env, task, pick_obj_location)
            robot_location = 'ready-pose'
        except Exception as e:
            print(f"[Error] Failed to go to ready pose: {e}")
            return

        # === Step 4: Move to drawer location if needed ===
        if robot_location != drawer_location:
            try:
                print(f"[Task] Moving robot from {robot_location} to {drawer_location} to place {pick_obj} in {drawer_name}")
                obs, reward, done = execute_go(env, task, robot_location, drawer_location)
                robot_location = drawer_location
            except Exception as e:
                print(f"[Error] Failed to move to drawer location: {e}")
                return

        # === Step 5: Place the object into the drawer ===
        try:
            print(f"[Task] Placing object {pick_obj} into drawer {drawer_name} at {drawer_location}")
            obs, reward, done = execute_place_to_drawer(env, task, pick_obj, drawer_name, drawer_location)
        except Exception as e:
            print(f"[Error] Failed to place object in drawer: {e}")
            return

        # === Step 6: Go to ready pose to clear (need-ready) again ===
        try:
            print(f"[Task] Going to ready pose to clear (need-ready) after placing")
            obs, reward, done = execute_go_ready(env, task, drawer_location)
            robot_location = 'ready-pose'
        except Exception as e:
            print(f"[Error] Failed to go to ready pose after placing: {e}")
            return

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
