# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (drawer-unlocked drawer1) ===
        # The feedback indicates that the predicate (drawer-unlocked drawer1) is missing.
        # We need to explore the environment to determine the lock state of drawer1.
        # We use only the available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Assumptions for object names and locations (update as needed for your environment)
        robot_location = None
        drawer1_location = None
        handle1_name = None
        handle1_location = None

        # Try to infer drawer1 and handle1 from positions
        for obj_name, pos in positions.items():
            if 'drawer1' in obj_name:
                drawer1_location = pos
            if 'handle' in obj_name and ('drawer1' in obj_name or '1' in obj_name):
                handle1_name = obj_name
                handle1_location = pos
            if 'robot' in obj_name or 'ready' in obj_name:
                robot_location = pos

        # Fallbacks if not found
        if robot_location is None:
            robot_location = positions[list(positions.keys())[0]]
        if drawer1_location is None:
            drawer1_location = positions[list(positions.keys())[0]]
        if handle1_name is None or handle1_location is None:
            for obj_name, pos in positions.items():
                if 'handle' in obj_name:
                    handle1_name = obj_name
                    handle1_location = pos
                    break

        # 1. Move robot to handle1 location if not already there
        try:
            print("[Exploration] Moving robot to handle1 location:", handle1_location)
            obs, reward, done = execute_go(
                env,
                task,
                from_pos=robot_location,
                to_pos=handle1_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            robot_location = handle1_location
        except Exception as e:
            print("[Exploration] Error during execute_go to handle1:", e)

        # 2. Pick up the handle (assume handle is on the floor)
        try:
            print("[Exploration] Picking up handle:", handle1_name)
            obs, reward, done = execute_pick(
                env,
                task,
                obj_name=handle1_name,
                obj_pos=handle1_location,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print("[Exploration] Error during execute_pick for handle:", e)

        # 3. Move to drawer1 location (if different from handle)
        if drawer1_location != handle1_location:
            try:
                print("[Exploration] Moving to drawer1 location:", drawer1_location)
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_pos=handle1_location,
                    to_pos=drawer1_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                robot_location = drawer1_location
            except Exception as e:
                print("[Exploration] Error during execute_go to drawer1:", e)

        # 4. Attempt to pull the drawer using the handle to determine lock state
        # This will only succeed if the drawer is unlocked; otherwise, it will fail or do nothing.
        try:
            print("[Exploration] Attempting to pull drawer1 with handle:", handle1_name)
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name='drawer1',
                handle_name=handle1_name,
                drawer_pos=drawer1_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            # If the drawer opens, we can infer (drawer-unlocked drawer1)
            print("[Exploration] Pull action executed. Check if drawer is open to infer lock state.")
        except Exception as e:
            print("[Exploration] Error during execute_pull for drawer1:", e)

        # 5. (Optional) Check state to confirm if drawer is now open (would require observation parsing)
        # For now, we assume the exploration phase is complete.

        print("[Exploration] Exploration phase complete. Missing predicate likely identified: (drawer-unlocked drawer1)")

        # === Main Task Plan Execution ===
        # At this point, you would continue with the oracle plan using the available skills.
        # For example, you might now pick/place objects, push/pull drawers, etc.
        # Insert your oracle plan steps here, using only the provided skill functions.

        # Example (commented out, as actual plan steps depend on your task):
        # obs, reward, done = execute_pick(env, task, obj_name='object1', obj_pos=positions['object1'], ...)
        # obs, reward, done = execute_place(env, task, obj_name='object1', target_pos=positions['bin'], ...)

        # TODO: Insert oracle plan steps here as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
