# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # According to the domain, after certain actions (e.g., pick/place), (need-ready) becomes true,
        # and the robot must execute 'execute_go_ready' to clear it before proceeding.
        # We will explore the environment to check if (need-ready) is set after a pick/place,
        # and if so, call the appropriate skill to clear it.

        # For demonstration, let's assume a simple plan:
        # 1. Move to an object on the floor.
        # 2. Pick the object (this will set (need-ready)).
        # 3. Try to pick another object (should fail if (need-ready) is not cleared).
        # 4. Call 'execute_go_ready' to clear (need-ready).
        # 5. Continue with the next action.

        # For this example, we will use the first two objects found in positions.
        # The actual plan should be replaced with the oracle plan for your task.

        # Find two objects on the floor (excluding handles)
        object_names = [name for name in positions if 'handle' not in name and 'drawer' not in name and 'bin' not in name]
        if len(object_names) < 1:
            print("[Exploration] No suitable objects found for pick action.")
            return

        obj1 = object_names[0]
        obj1_pos = positions[obj1]

        # Find robot's initial location
        robot_location = None
        for name, pos in positions.items():
            if 'robot' in name:
                robot_location = pos
                break

        # If robot location is not found, use a default or the first location
        if robot_location is None:
            robot_location = list(positions.values())[0]

        # 1. Move to the object's location
        try:
            print(f"[Exploration] Moving to {obj1} at {obj1_pos}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=obj1_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            robot_location = obj1_pos
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")
            return

        # 2. Pick the object (this should set (need-ready))
        try:
            print(f"[Exploration] Picking {obj1} at {obj1_pos}")
            obs, reward, done = execute_pick(
                env,
                task,
                target_object=obj1,
                target_pos=obj1_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Exploration] Error during execute_pick: {e}")
            return

        # 3. Try to pick another object (should fail if (need-ready) is not cleared)
        if len(object_names) > 1:
            obj2 = object_names[1]
            obj2_pos = positions[obj2]
            try:
                print(f"[Exploration] Attempting to pick {obj2} at {obj2_pos} (should fail if need-ready is set)")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target_object=obj2,
                    target_pos=obj2_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print("[Exploration] Unexpectedly succeeded in picking without clearing need-ready.")
            except Exception as e:
                print(f"[Exploration] As expected, could not pick {obj2} due to need-ready: {e}")

        # 4. Call 'execute_go_ready' to clear (need-ready)
        try:
            print("[Exploration] Executing go_ready to clear need-ready predicate.")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=robot_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            # After this, (need-ready) should be cleared
        except Exception as e:
            print(f"[Exploration] Error during execute_go_ready: {e}")
            return

        # 5. Try to pick the second object again (should now succeed)
        if len(object_names) > 1:
            obj2 = object_names[1]
            obj2_pos = positions[obj2]
            try:
                print(f"[Exploration] Now picking {obj2} at {obj2_pos} after clearing need-ready.")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target_object=obj2,
                    target_pos=obj2_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print("[Exploration] Successfully picked second object after clearing need-ready.")
            except Exception as e:
                print(f"[Exploration] Error during second pick: {e}")

        print("[Exploration] Exploration phase complete. The missing predicate is (need-ready).")
        print("[Exploration] The plan must include execute_go_ready after actions that set (need-ready).")

        # === End of Exploration Phase ===

        # TODO: Replace the above with the actual oracle plan for your task,
        # ensuring that after any action that sets (need-ready), you call execute_go_ready
        # before attempting further actions that require (not (need-ready)).

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
