# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (drawer-unlocked drawer1) ===
        # The feedback indicates that the predicate (drawer-unlocked drawer1) is missing.
        # We need to explore the environment to determine the lock state of drawer1.
        # We use only the available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Assumptions:
        # - There is a drawer named 'drawer1' and possibly a handle object for it.
        # - The robot must determine if 'drawer1' is unlocked by attempting to pull it.
        # - The robot may need to move to the drawer's location and pick the handle before pulling.

        # Step 1: Get relevant object and location names
        # These should be available from the environment or positions dictionary
        # For this example, we assume the following keys exist:
        #   - 'drawer1': the drawer object
        #   - 'handle1': the handle object for drawer1
        #   - 'drawer1_location': the location of drawer1/handle1
        #   - 'ready-pose': the robot's ready position

        try:
            drawer_name = 'drawer1'
            handle_name = 'handle1'
            drawer_location = positions.get('drawer1_location', None)
            handle_location = positions.get('handle1_location', drawer_location)
            ready_pose = positions.get('ready-pose', None)
        except Exception as e:
            print("[Error] Could not retrieve object positions:", e)
            shutdown_environment(env)
            return

        # Step 2: Move to handle location if not already there
        try:
            # Get current robot location from observation or positions
            robot_location = positions.get('robot', {}).get('location', None)
            if robot_location is None:
                # Fallback: try to get from obs or assume ready-pose
                robot_location = ready_pose

            if robot_location != handle_location:
                print(f"[Exploration] Moving robot from {robot_location} to handle at {handle_location}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=handle_location
                )
                robot_location = handle_location
        except Exception as e:
            print("[Error] Failed to move to handle location:", e)
            shutdown_environment(env)
            return

        # Step 3: Pick the handle (if not already holding)
        try:
            print(f"[Exploration] Attempting to pick handle '{handle_name}' at {handle_location}")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=handle_name,
                location=handle_location
            )
        except Exception as e:
            print("[Error] Failed to pick handle:", e)
            shutdown_environment(env)
            return

        # Step 4: Attempt to pull the drawer using the handle
        try:
            print(f"[Exploration] Attempting to pull drawer '{drawer_name}' with handle '{handle_name}' at {drawer_location}")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name=handle_name,
                location=drawer_location
            )
            # If the pull succeeds, we can infer the drawer is unlocked
            print("[Exploration] Pull action executed. If drawer opened, (drawer-unlocked drawer1) is true.")
        except Exception as e:
            print("[Exploration] Pull action failed. Drawer may be locked or another issue occurred.")
            print("[Error]", e)
            # The failure to pull may indicate the drawer is locked

        # Step 5: (Optional) Return to ready pose
        try:
            if ready_pose is not None and robot_location != ready_pose:
                print(f"[Exploration] Returning robot to ready pose at {ready_pose}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=ready_pose
                )
        except Exception as e:
            print("[Error] Failed to return to ready pose:", e)

        print("===== Exploration Phase Complete =====")
        print("If the drawer was successfully pulled open, the missing predicate is (drawer-unlocked drawer1).")
        print("Otherwise, further exploration or unlocking is required.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
