# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # Feedback indicates (need-ready) is a missing predicate that blocks further actions.
        # We must ensure that after any action that sets (need-ready), we call the appropriate skill to clear it.
        # In the provided domain, the only action that clears (need-ready) is execute_go_ready.
        # So, after any action that sets (need-ready), we must call execute_go_ready before proceeding.

        # For demonstration, let's assume a generic plan:
        # 1. Pick up an object from the floor (execute_pick)
        # 2. Go to ready pose (execute_go_ready) to clear (need-ready)
        # 3. Move to a drawer (execute_go)
        # 4. Place the object into the drawer (execute_place)
        # 5. Go to ready pose (execute_go_ready) to clear (need-ready)
        # 6. Push the drawer closed (execute_push)
        # (The actual plan may differ, but this illustrates the need to handle (need-ready))

        # === Example: Find an object on the floor and a drawer ===
        # For this example, we will search for an object and a drawer in the positions dict.
        object_name = None
        object_pos = None
        drawer_name = None
        drawer_pos = None
        for name, pos in positions.items():
            if 'drawer' in name and drawer_name is None:
                drawer_name = name
                drawer_pos = pos
            elif 'handle' not in name and object_name is None:
                object_name = name
                object_pos = pos
        if object_name is None or drawer_name is None:
            print("[Error] Could not find both an object and a drawer in the environment.")
            return

        # 1. Pick up the object from the floor
        print(f"[Task] Picking up object '{object_name}' at {object_pos}")
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                obj_name=object_name,
                location=object_pos
            )
        except Exception as e:
            print(f"[Error] Failed to pick object: {e}")
            return

        # 2. Go to ready pose to clear (need-ready)
        print("[Task] Going to ready pose to clear (need-ready)")
        try:
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=object_pos
            )
        except Exception as e:
            print(f"[Error] Failed to go to ready pose: {e}")
            return

        # 3. Move to the drawer
        print(f"[Task] Moving to drawer '{drawer_name}' at {drawer_pos}")
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location='ready-pose',
                to_location=drawer_pos
            )
        except Exception as e:
            print(f"[Error] Failed to move to drawer: {e}")
            return

        # 4. Place the object into the drawer
        print(f"[Task] Placing object '{object_name}' into drawer '{drawer_name}'")
        try:
            obs, reward, done = execute_place(
                env,
                task,
                obj_name=object_name,
                drawer_name=drawer_name,
                location=drawer_pos
            )
        except Exception as e:
            print(f"[Error] Failed to place object into drawer: {e}")
            return

        # 5. Go to ready pose to clear (need-ready)
        print("[Task] Going to ready pose to clear (need-ready) after placing")
        try:
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=drawer_pos
            )
        except Exception as e:
            print(f"[Error] Failed to go to ready pose after placing: {e}")
            return

        # 6. Push the drawer closed
        print(f"[Task] Pushing drawer '{drawer_name}' closed")
        try:
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=drawer_name,
                location=drawer_pos
            )
        except Exception as e:
            print(f"[Error] Failed to push drawer closed: {e}")
            return

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
