# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that (not (need-ready)) is a missing predicate in preconditions.
        # We will attempt to execute a skill that requires (not (need-ready)) and handle failure.
        # If the action fails, we infer that (need-ready) is True and we must execute the ready action.

        # For demonstration, we will attempt to execute a simple skill (e.g., execute_go) between two locations.
        # If it fails due to (need-ready), we will execute execute_go_ready to clear the need-ready flag.

        # Find two locations from positions (assuming at least two exist)
        location_names = [name for name, pos in positions.items() if 'location' in name or 'pose' in name or 'drawer' in name or 'bin' in name]
        if len(location_names) < 2:
            # Fallback: just use any two keys
            location_names = list(positions.keys())[:2]
        if len(location_names) < 2:
            print("[Exploration] Not enough locations found in positions to perform exploration.")
            return

        from_location = location_names[0]
        to_location = location_names[1]

        print(f"[Exploration] Attempting to move from {from_location} to {to_location} using execute_go.")

        # Try to execute_go; if it fails due to need-ready, execute execute_go_ready and try again
        exploration_success = False
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location,
                to_location
            )
            print(f"[Exploration] execute_go succeeded from {from_location} to {to_location}.")
            exploration_success = True
        except Exception as e:
            print(f"[Exploration] execute_go failed: {e}")
            # Assume failure is due to (need-ready) being True
            print("[Exploration] Detected missing predicate: (not (need-ready)). Executing execute_go_ready to clear need-ready.")
            try:
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location
                )
                print("[Exploration] execute_go_ready succeeded. Retrying execute_go.")
                # Retry execute_go
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location,
                    to_location
                )
                print(f"[Exploration] execute_go succeeded after clearing need-ready from {from_location} to {to_location}.")
                exploration_success = True
            except Exception as e2:
                print(f"[Exploration] execute_go_ready or retry execute_go failed: {e2}")
                exploration_success = False

        if exploration_success:
            print("[Exploration] Exploration phase completed: (need-ready) predicate identified and handled.")
        else:
            print("[Exploration] Exploration phase failed: could not resolve (need-ready) predicate.")

        # === Continue with Task Plan (if any) ===
        # At this point, you can proceed with the rest of your oracle plan, always checking for (need-ready)
        # before executing actions that require (not (need-ready)) in their preconditions.
        # For each such action, if it fails, call execute_go_ready first.

        # Example: (pseudo-code, not executed here)
        # for step in oracle_plan:
        #     try:
        #         obs, reward, done = skill_fn(env, task, *args)
        #     except Exception as e:
        #         if "need-ready" in str(e):
        #             execute_go_ready(env, task, current_location)
        #             obs, reward, done = skill_fn(env, task, *args)
        #         else:
        #             raise

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
