# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (drawer-unlocked drawer1) ===
        # The feedback indicates that the predicate (drawer-unlocked drawer1) is missing.
        # We need to explore and check the lock state of drawer1 before attempting to pull/open it.

        # For this, we use the available skills to try to pull the drawer and observe the result.
        # If the drawer is locked, the pull will fail or have no effect; if unlocked, it will succeed.

        # Assume object and location names based on typical conventions
        drawer_name = 'drawer1'
        handle_name = 'handle1'
        robot_location = 'ready-pose'
        drawer_location = 'drawer1_location'
        handle_location = 'handle1_location'

        # Try to get actual positions if available
        if 'drawer1' in positions:
            drawer_pos = positions['drawer1']
        else:
            drawer_pos = None
        if 'handle1' in positions:
            handle_pos = positions['handle1']
        else:
            handle_pos = None

        # Step 1: Move robot to handle location if not already there
        try:
            print("[Exploration] Moving robot to handle location.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=handle_location,
                max_steps=100
            )
        except Exception as e:
            print(f"[Exploration] Error during execute_go to handle: {e}")

        # Step 2: Pick up the handle
        try:
            print("[Exploration] Attempting to pick up the handle.")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=handle_name,
                location=handle_location,
                max_steps=100
            )
        except Exception as e:
            print(f"[Exploration] Error during execute_pick handle: {e}")

        # Step 3: Move to drawer location (if different)
        try:
            print("[Exploration] Moving robot to drawer location.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=handle_location,
                to_location=drawer_location,
                max_steps=100
            )
        except Exception as e:
            print(f"[Exploration] Error during execute_go to drawer: {e}")

        # Step 4: Try to pull the drawer (this will help us determine if it's locked/unlocked)
        try:
            print("[Exploration] Attempting to pull the drawer to check lock state.")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name=handle_name,
                location=drawer_location,
                max_steps=100
            )
            # If the pull succeeds, we can infer the drawer is unlocked.
            print("[Exploration] Pull action executed. If drawer opened, it is unlocked.")
        except Exception as e:
            print(f"[Exploration] Error during execute_pull: {e}")
            print("[Exploration] Drawer may be locked or predicate missing.")

        # === Main Task Plan (After Exploration) ===
        # At this point, we have explored the lock state.
        # If the drawer is unlocked, proceed with the main plan (e.g., pick/place objects).
        # If not, handle accordingly (could log, retry, or abort).

        # Example: Place an object into the drawer if it is open
        # (Replace 'object1' and locations with actual names as needed)
        object_to_place = 'object1'
        object_location = 'object1_location'

        try:
            print("[Task] Moving to object location.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=drawer_location,
                to_location=object_location,
                max_steps=100
            )
        except Exception as e:
            print(f"[Task] Error during execute_go to object: {e}")

        try:
            print("[Task] Picking up the object.")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=object_to_place,
                location=object_location,
                max_steps=100
            )
        except Exception as e:
            print(f"[Task] Error during execute_pick object: {e}")

        try:
            print("[Task] Moving to drawer location to place object.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=object_location,
                to_location=drawer_location,
                max_steps=100
            )
        except Exception as e:
            print(f"[Task] Error during execute_go to drawer for placing: {e}")

        try:
            print("[Task] Placing object into the drawer.")
            obs, reward, done = execute_place(
                env,
                task,
                object_name=object_to_place,
                location=drawer_location,
                max_steps=100
            )
        except Exception as e:
            print(f"[Task] Error during execute_place: {e}")

        # Optionally, push the drawer closed
        try:
            print("[Task] Pushing the drawer closed.")
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=drawer_name,
                location=drawer_location,
                max_steps=100
            )
        except Exception as e:
            print(f"[Task] Error during execute_push: {e}")

        # Return to ready pose
        try:
            print("[Task] Returning to ready pose.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=drawer_location,
                to_location='ready-pose',
                max_steps=100
            )
        except Exception as e:
            print(f"[Task] Error during execute_go to ready-pose: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()