# run_skeleton_task.py (Fully Generic Skeleton, WITH EXPLORATION/DIAGNOSTICS PHASE)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Provided skills; DO NOT redefine any primitives

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset task to initial state and get objects/initial obs
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers
        init_video_writers(obs)

        # Wrap task steps for video/obs recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Get object positions etc., if needed
        positions = get_object_positions()
        
        # --- EXPLORATION: Find/Diagnose Missing Predicates via Exploration Phase ---

        # Problem Knowledge: Feedback indicates (holding handle_top) is missing,
        # so the issue is likely not being able to recognize/predict holding correctly,
        # or handle-specific exploration (e.g., lock-known status, as per exploration domain).

        # For this diagnostic/exploration phase, we try to execute the relevant skills 
        # and see if missing predicates (such as lock-known) are properly identified.

        # We attempt to identify ALL potentially missing/unknown facts for handle objects.

        # (For demonstration purposes, you can adapt object/handle names here as needed)
        # Example: assume in observation / object_positions / description there is a handle_top
        # The actual names of handles/objects/locations may need to be adapted to your scenario.
        # The code below is written to be general, you may need to map 'handle_top' to an actual object.

        handle_obj_name = "handle_top"   # Replace with actual handle object name if different

        # 1. Ensure we know the pose of the handle and the robot's current location
        handle_pos = positions.get(handle_obj_name, None)
        if handle_pos is None:
            print(f"[Exploration] Could not find position for '{handle_obj_name}'. Skipping exploration/diagnostic phase.")
        else:
            print(f"[Exploration] Found pose for '{handle_obj_name}':", handle_pos)
        
            # 2. Try picking up the handle (if it is on the floor and not currently held)
            # The exact call depends on the skill_code interface. 
            # Since 'execute_pick' is a generic name, we try to use it.
            # Some domains may separate pick_handle from pick_object, but both are available in PDDL.
            try:
                obs, reward, done = execute_pick(
                    env, task,
                    target=handle_obj_name,
                    location=handle_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Attempted execute_pick on '{handle_obj_name}'.")
            except Exception as e:
                print(f"[Exploration] Exception during execute_pick: {e}")

            # 3. Try any *diagnostic* skills related to lock/durability/weight-known etc,
            # if such a skill is available in skill_code (not specified in this domain),
            # else try generic tasks for exploration (move to, interact, pull, etc).

            # Example: execute_pull may trigger lock-known for a handle/drawer.
            try:
                # If the domain uses drawer/handle relationships, 
                # identify corresponding drawer for the handle, if possible
                handle_drawer = None
                for obj, pos in positions.items():
                    if "drawer" in obj:
                        handle_drawer = obj
                        break
                if handle_drawer is not None:
                    # This assumes you have a pull skill for a handle and its drawer
                    obs, reward, done = execute_pull(
                        env, task,
                        drawer=handle_drawer,
                        handle=handle_obj_name,
                        location=handle_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    print(f"[Exploration] Attempted execute_pull on '{handle_obj_name}' with drawer '{handle_drawer}'.")
                else:
                    print(f"[Exploration] No drawer object detected, skipping execute_pull for handle.")
            except Exception as e:
                print(f"[Exploration] Exception during execute_pull: {e}")

            # If further exploration skills (e.g., weight/durability identification) were provided,
            # they would be invoked here, but your only available skills at present are those listed.

        # --- END Exploration Phase ---
        print("[Exploration] Completed exploration phase. Review logs above for missing predicate diagnosis.")

        # ----------------- TASK EXECUTION PHASE ---------------- 
        # Here, you would execute the oracle (plan) for your problem  
        # Plan execution should call only the available skills with correct objects and poses.

        # For demonstration, suppose the plan requires:
        # (1) Move robot to location of handle_top
        # (2) Pick handle_top
        # (3) Pull the drawer using handle_top
        # (4) etc.
        # Adapt as needed based on your domain/problem instance!

        # 1. Move robot to handle
        try:
            obs, reward, done = execute_go(
                env, task,
                from_location="ready-pose",
                to_location=handle_pos,   # Use location string if appropriate, else use its coordinate
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print("[Task] Moved robot to handle location.")
        except Exception as e:
            print(f"[Task] Exception during execute_go: {e}")

        # 2. Pick the handle
        try:
            obs, reward, done = execute_pick(
                env, task,
                target=handle_obj_name,
                location=handle_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            print("[Task] Picked the handle.")
        except Exception as e:
            print(f"[Task] Exception during execute_pick (task phase): {e}")

        # 3. Pull the drawer (assuming we have the drawer name as above)
        try:
            if handle_drawer is not None:
                obs, reward, done = execute_pull(
                    env, task,
                    drawer=handle_drawer,
                    handle=handle_obj_name,
                    location=handle_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print("[Task] Pulled the drawer open with the handle.")
            else:
                print(f"[Task] No drawer found for pulling. Skipping execute_pull.")
        except Exception as e:
            print(f"[Task] Exception during execute_pull (task phase): {e}")

        # 4. Continue with further plan steps (pick objects, place, push, etc.)
        #   Insert appropriate calls to available skills here, 
        #   using observations, positions, and plan logic.
        #   e.g.,
        # obs, reward, done = execute_pick(...)
        # obs, reward, done = execute_place(...)
        # obs, reward, done = execute_push(...)
        # etc.

        # For each, always check `done`, `reward`, or observation as needed.
        # The actual sequence depends on the intended goal/task/oracle plan supplied for your scenario.

        print("[Task] Plan execution completed (or as far as implemented).")

    finally:
        # Properly shutdown the environment no matter what
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
