# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Try to Identify Missing Predicate ===
        # The feedback suggests a missing predicate is blocking planning.
        # We'll use available skills to explore the environment and try to trigger all possible predicates.
        # This will help us identify which predicate is missing or required.

        # For demonstration, we attempt to use all available skills in a safe, exploratory way.
        # This is a generic exploration; in a real scenario, you would tailor this to your domain and objects.

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Get all objects and locations from positions
        object_names = [name for name in positions if not name.startswith('robot') and not name.startswith('drawer')]
        location_names = [name for name in positions if name.startswith('location') or name.startswith('drawer') or name == 'bin' or name == 'ready-pose']

        # Try to move the robot to each location (execute_go)
        current_location = None
        try:
            # Try to get robot's current location
            for name in positions:
                if name.startswith('robot'):
                    current_location = positions[name]
                    break
        except Exception as e:
            print("[Exploration] Could not determine robot's current location:", e)

        # If we don't know the robot's location, just pick the first location as a starting point
        if current_location is None and location_names:
            current_location = location_names[0]

        # Try moving to each location
        for loc in location_names:
            try:
                print(f"[Exploration] Trying to move to location: {loc}")
                obs, reward, done = execute_go(env, task, from_location=current_location, to_location=loc)
                current_location = loc
                if done:
                    print("[Exploration] Task ended during exploration (move)!")
                    return
            except Exception as e:
                print(f"[Exploration] execute_go failed for {loc}: {e}")

        # Try picking up each object (execute_pick)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to pick object: {obj} at location: {loc}")
                    obs, reward, done = execute_pick(env, task, object_name=obj, location=loc)
                    if done:
                        print("[Exploration] Task ended during exploration (pick)!")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_pick failed for {obj} at {loc}: {e}")

        # Try placing each object (execute_place)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to place object: {obj} at location: {loc}")
                    obs, reward, done = execute_place(env, task, object_name=obj, location=loc)
                    if done:
                        print("[Exploration] Task ended during exploration (place)!")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_place failed for {obj} at {loc}: {e}")

        # Try pushing and pulling drawers (execute_push, execute_pull)
        for drawer in [name for name in positions if name.startswith('drawer')]:
            for obj in object_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Trying to pull drawer: {drawer} with object: {obj} at location: {loc}")
                        obs, reward, done = execute_pull(env, task, drawer_name=drawer, handle_object=obj, location=loc)
                        if done:
                            print("[Exploration] Task ended during exploration (pull)!")
                            return
                    except Exception as e:
                        print(f"[Exploration] execute_pull failed for {drawer} with {obj} at {loc}: {e}")
                    try:
                        print(f"[Exploration] Trying to push drawer: {drawer} at location: {loc}")
                        obs, reward, done = execute_push(env, task, drawer_name=drawer, location=loc)
                        if done:
                            print("[Exploration] Task ended during exploration (push)!")
                            return
                    except Exception as e:
                        print(f"[Exploration] execute_push failed for {drawer} at {loc}: {e}")

        # Try sweeping and rotating objects (execute_sweep, execute_rotate)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to sweep object: {obj} at location: {loc}")
                    obs, reward, done = execute_sweep(env, task, object_name=obj, location=loc)
                    if done:
                        print("[Exploration] Task ended during exploration (sweep)!")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed for {obj} at {loc}: {e}")
                try:
                    print(f"[Exploration] Trying to rotate object: {obj} at location: {loc}")
                    obs, reward, done = execute_rotate(env, task, object_name=obj, location=loc)
                    if done:
                        print("[Exploration] Task ended during exploration (rotate)!")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_rotate failed for {obj} at {loc}: {e}")

        # Try gripper action (execute_gripper)
        try:
            print("[Exploration] Trying gripper action")
            obs, reward, done = execute_gripper(env, task)
            if done:
                print("[Exploration] Task ended during exploration (gripper)!")
                return
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Exploration phase completed. Check logs for any missing predicate clues.")

        # === End of Exploration Phase ===

        # === (Optional) Main Task Plan Execution ===
        # At this point, after exploration, you would normally execute the oracle plan.
        # For this template, the focus is on exploration for missing predicates.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()