# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a planning failure, likely due to a missing predicate.
        # We use the exploration domain knowledge to try to discover which predicate is missing.
        # We attempt to use available skills to interact with objects and locations, and observe the effects.

        # For demonstration, we will:
        # 1. Move the robot to each location and try to identify objects (simulate 'identified' predicate)
        # 2. Attempt to pick objects and see if any information (e.g., weight, durability) is revealed
        # 3. Attempt to pull objects (e.g., handles) to check for lock state knowledge

        # Note: Only use available skills: 
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Get all objects and locations from positions
        object_names = [name for name in positions if not name.startswith('location_')]
        location_names = [name for name in positions if name.startswith('location_')]

        # If location names are not in the form 'location_x', try to infer from positions
        if not location_names:
            # Try to infer locations from object positions
            location_names = list(set([positions[obj]['location'] for obj in positions if 'location' in positions[obj]]))

        # 1. Exploration: Move to each location and try to identify objects
        for loc in location_names:
            try:
                # Get current robot location (assume 'robot' key in positions or from obs)
                robot_loc = None
                if 'robot' in positions and 'location' in positions['robot']:
                    robot_loc = positions['robot']['location']
                else:
                    # Fallback: try to get from obs or assume first location
                    robot_loc = location_names[0]
                if robot_loc == loc:
                    continue  # Already at location

                print(f"[Exploration] Moving robot from {robot_loc} to {loc} to identify objects.")
                obs, reward, done = execute_go(env, task, from_location=robot_loc, to_location=loc)
                if done:
                    print("[Exploration] Task ended during exploration move.")
                    return
                # Update robot location
                robot_loc = loc

                # Try to sweep (simulate identification) for each object at this location
                for obj in object_names:
                    obj_info = positions[obj]
                    if 'location' in obj_info and obj_info['location'] == loc:
                        print(f"[Exploration] Sweeping object {obj} at {loc} to identify.")
                        try:
                            obs, reward, done = execute_sweep(env, task, obj, loc)
                            if done:
                                print("[Exploration] Task ended during sweep.")
                                return
                        except Exception as e:
                            print(f"[Exploration] Sweep failed for {obj} at {loc}: {e}")

            except Exception as e:
                print(f"[Exploration] Move failed to {loc}: {e}")

        # 2. Exploration: Try to pick each object and see if any new info is revealed
        for obj in object_names:
            obj_info = positions[obj]
            if 'location' in obj_info:
                loc = obj_info['location']
                print(f"[Exploration] Attempting to pick object {obj} at {loc}.")
                try:
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    if done:
                        print("[Exploration] Task ended during pick.")
                        return
                    # Try to place it back (if possible)
                    print(f"[Exploration] Attempting to place object {obj} at {loc}.")
                    try:
                        obs, reward, done = execute_place(env, task, obj, loc)
                        if done:
                            print("[Exploration] Task ended during place.")
                            return
                    except Exception as e:
                        print(f"[Exploration] Place failed for {obj} at {loc}: {e}")
                except Exception as e:
                    print(f"[Exploration] Pick failed for {obj} at {loc}: {e}")

        # 3. Exploration: Try to pull handles to check for lock state knowledge
        for obj in object_names:
            obj_info = positions[obj]
            if obj_info.get('is_handle', False):
                loc = obj_info['location']
                print(f"[Exploration] Attempting to pull handle {obj} at {loc}.")
                try:
                    obs, reward, done = execute_pull(env, task, obj, loc)
                    if done:
                        print("[Exploration] Task ended during pull.")
                        return
                except Exception as e:
                    print(f"[Exploration] Pull failed for {obj} at {loc}: {e}")

        # 4. Exploration: Try to push drawers (if any) to check for state changes
        for obj in object_names:
            obj_info = positions[obj]
            if obj_info.get('is_drawer', False):
                loc = obj_info['location']
                print(f"[Exploration] Attempting to push drawer {obj} at {loc}.")
                try:
                    obs, reward, done = execute_push(env, task, obj, loc)
                    if done:
                        print("[Exploration] Task ended during push.")
                        return
                except Exception as e:
                    print(f"[Exploration] Push failed for {obj} at {loc}: {e}")

        # 5. Exploration: Try gripper action (if available)
        try:
            print("[Exploration] Attempting gripper action.")
            obs, reward, done = execute_gripper(env, task)
            if done:
                print("[Exploration] Task ended during gripper action.")
                return
        except Exception as e:
            print(f"[Exploration] Gripper action failed: {e}")

        print("[Exploration] Exploration phase completed. Check logs and observations for missing predicate clues.")

        # === End of Exploration Phase ===

        # TODO: After exploration, analyze observations/logs to determine which predicate is missing.
        # This may require human-in-the-loop or automated log analysis (not implemented here).

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()