# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Based on feedback, the planner times out, likely due to a missing predicate in the domain/problem.
        # We will use the available skills to explore the environment and try to identify which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects and see if any expected state change is missing.

        # For demonstration, we will try to:
        # 1. Move to each location
        # 2. Attempt to pick up objects
        # 3. Attempt to pull/push drawers/objects
        # 4. Log any unexpected failures or missing state transitions

        # Get all objects and locations from positions
        object_names = [name for name in positions if not name.startswith('location_')]
        location_names = [name for name in positions if name.startswith('location_')]

        # For this example, we assume the robot starts at a known location
        # Try to move to each location and interact with objects
        current_location = None
        for loc in location_names:
            try:
                # Move to location if not already there
                if current_location is not None and loc != current_location:
                    print(f"[Exploration] Moving from {current_location} to {loc}")
                    obs, reward, done = execute_go(env, task, from_location=current_location, to_location=loc)
                    if done:
                        print("[Exploration] Task ended unexpectedly during move.")
                        return
                current_location = loc

                # Identify objects at this location
                objects_here = [obj for obj in object_names if positions[obj] == positions[loc]]
                for obj in objects_here:
                    print(f"[Exploration] At {loc}, found object {obj}")
                    # Try to pick the object
                    try:
                        print(f"[Exploration] Attempting to pick {obj} at {loc}")
                        obs, reward, done = execute_pick(env, task, object_name=obj, location=loc)
                        if done:
                            print("[Exploration] Task ended after pick.")
                            return
                    except Exception as e:
                        print(f"[Exploration] Could not pick {obj} at {loc}: {e}")

                    # Try to pull the object (if applicable)
                    try:
                        print(f"[Exploration] Attempting to pull {obj} at {loc}")
                        obs, reward, done = execute_pull(env, task, object_name=obj, location=loc)
                        if done:
                            print("[Exploration] Task ended after pull.")
                            return
                    except Exception as e:
                        print(f"[Exploration] Could not pull {obj} at {loc}: {e}")

                    # Try to push the object (if applicable)
                    try:
                        print(f"[Exploration] Attempting to push {obj} at {loc}")
                        obs, reward, done = execute_push(env, task, object_name=obj, location=loc)
                        if done:
                            print("[Exploration] Task ended after push.")
                            return
                    except Exception as e:
                        print(f"[Exploration] Could not push {obj} at {loc}: {e}")

            except Exception as e:
                print(f"[Exploration] Error during exploration at {loc}: {e}")

        # After exploration, print a summary
        print("[Exploration] Exploration phase complete. Check logs for missing predicates or unexpected failures.")

        # === Main Task Plan Execution (Placeholder) ===
        # Here you would execute the oracle plan step-by-step using the available skills.
        # For example:
        # obs, reward, done = execute_pick(env, task, object_name='obj1', location='location_a')
        # obs, reward, done = execute_place(env, task, object_name='obj1', location='location_b')
        # ... etc.
        # If the plan is available, iterate through it and call the corresponding skill functions.

        # TODO: Insert oracle plan execution here as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
