# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (drawer-unlocked drawer1) is missing.
        # We need to explore and check the lock state of drawer1.
        # The exploration domain provides an 'execute_pull' action to determine lock-known.
        # In our available skills, we have 'execute_pull' (for pulling a drawer open).
        # We will attempt to pull the drawer handle and observe if the drawer is locked or unlocked.

        # --- Step 1: Move to the drawer's location ---
        # Assume 'drawer1' and its handle are present in positions
        # and that the robot starts at 'ready-pose' or initial location.
        # We'll use 'execute_go' to move to the drawer location.

        # Find drawer1 and its handle
        drawer_name = 'drawer1'
        handle_name = None
        for obj in positions:
            if 'handle' in obj and drawer_name in obj:
                handle_name = obj
                break
        if handle_name is None:
            # Fallback: pick any handle
            for obj in positions:
                if 'handle' in obj:
                    handle_name = obj
                    break

        if handle_name is None:
            print("[Exploration] No handle found for drawer1. Exploration cannot proceed.")
            return

        # Get locations
        try:
            drawer_pos = positions[drawer_name]
            handle_pos = positions[handle_name]
        except KeyError:
            print("[Exploration] Drawer or handle position not found in positions dictionary.")
            return

        # Assume location names are available (e.g., 'drawer_area', 'ready-pose')
        # For this example, we use the key names as location identifiers
        robot_location = 'ready-pose'
        drawer_location = drawer_name  # or a specific location if available

        # --- Step 2: Move to the drawer location ---
        try:
            print(f"[Exploration] Moving robot from {robot_location} to {drawer_location}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=drawer_location
            )
            robot_location = drawer_location
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")
            return

        # --- Step 3: Pick up the handle (if not already holding) ---
        try:
            print(f"[Exploration] Picking up handle {handle_name} at {drawer_location}")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=handle_name,
                location=drawer_location
            )
        except Exception as e:
            print(f"[Exploration] Error during execute_pick: {e}")
            return

        # --- Step 4: Attempt to pull the drawer (execute_pull) ---
        # This will both open the drawer (if unlocked) and, in the exploration domain, set lock-known
        try:
            print(f"[Exploration] Attempting to pull drawer {drawer_name} using handle {handle_name} at {drawer_location}")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name=handle_name,
                location=drawer_location
            )
            print("[Exploration] Pull action executed. If drawer was unlocked, it should now be open.")
        except Exception as e:
            print(f"[Exploration] Error during execute_pull: {e}")
            return

        # --- Step 5: Check if the drawer is now open (i.e., unlocked) ---
        # In a real system, we would check the state via observation or predicates.
        # For this skeleton, we assume the feedback (drawer-unlocked drawer1) is now known.

        print("[Exploration] Exploration phase complete. Predicate (drawer-unlocked drawer1) is now known.")

        # === Main Task Plan Execution ===
        # Now that we know the drawer is unlocked, proceed with the oracle plan.
        # Example: pick an object from the floor, place it in the drawer, close the drawer, return to ready-pose.

        # --- Step 6: Pick an object from the floor ---
        # Find an object on the floor (not a handle)
        object_to_pick = None
        for obj in positions:
            if obj != handle_name and obj != drawer_name and 'handle' not in obj and 'drawer' not in obj:
                object_to_pick = obj
                break
        if object_to_pick is None:
            print("[Task] No suitable object found to pick from the floor.")
            return

        try:
            print(f"[Task] Picking up object {object_to_pick} from floor at {robot_location}")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=object_to_pick,
                location=robot_location
            )
        except Exception as e:
            print(f"[Task] Error during execute_pick: {e}")
            return

        # --- Step 7: Place the object into the drawer ---
        try:
            print(f"[Task] Placing object {object_to_pick} into drawer {drawer_name} at {drawer_location}")
            obs, reward, done = execute_place(
                env,
                task,
                object_name=object_to_pick,
                drawer_name=drawer_name,
                location=drawer_location
            )
        except Exception as e:
            print(f"[Task] Error during execute_place: {e}")
            return

        # --- Step 8: Push the drawer closed ---
        try:
            print(f"[Task] Pushing drawer {drawer_name} closed at {drawer_location}")
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=drawer_name,
                location=drawer_location
            )
        except Exception as e:
            print(f"[Task] Error during execute_push: {e}")
            return

        # --- Step 9: Return to ready-pose ---
        try:
            print(f"[Task] Moving robot from {robot_location} to ready-pose")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location='ready-pose'
            )
            robot_location = 'ready-pose'
        except Exception as e:
            print(f"[Task] Error during execute_go to ready-pose: {e}")
            return

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
