# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # In the domain, many actions set (need-ready) as an effect, and require (not (need-ready)) as a precondition.
        # Therefore, after any action that sets (need-ready), we must call execute_go_ready to clear it.

        # For demonstration, let's assume a generic plan:
        # 1. Move to an object location
        # 2. Pick the object (sets need-ready)
        # 3. Go to ready pose (clears need-ready)
        # 4. Move to a drawer
        # 5. Place the object in the drawer (sets need-ready)
        # 6. Go to ready pose (clears need-ready)
        # 7. Push the drawer closed (does not set need-ready)
        # (This is a generic plan; in practice, you would parse the oracle plan.)

        # For this example, let's use placeholder names for objects and locations.
        # Replace these with actual names from your environment as needed.
        object_name = None
        object_pos = None
        drawer_name = None
        drawer_pos = None
        ready_pose = 'ready-pose'
        robot_pos = None

        # Try to find an object and a drawer from positions
        for name, pos in positions.items():
            if 'drawer' in name and drawer_name is None:
                drawer_name = name
                drawer_pos = pos
            elif object_name is None:
                object_name = name
                object_pos = pos

        # If not found, abort
        if object_name is None or drawer_name is None:
            print("[Error] Could not find both an object and a drawer in the environment.")
            return

        # Assume robot starts at some initial position
        robot_pos = positions.get('robot', None)
        if robot_pos is None:
            # Fallback: use a default location
            robot_pos = (0, 0, 0)

        # === Step 1: Move to object location ===
        print(f"[Task] Moving to object '{object_name}' at {object_pos}")
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_pos,
                to_location=object_pos,
                timeout=10.0
            )
            robot_pos = object_pos
        except Exception as e:
            print(f"[Error] Failed to move to object: {e}")
            return

        # === Step 2: Pick the object (sets need-ready) ===
        print(f"[Task] Picking object '{object_name}'")
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=object_name,
                location=object_pos,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Failed to pick object: {e}")
            return

        # === Step 3: Go to ready pose (clears need-ready) ===
        print(f"[Task] Going to ready pose to clear (need-ready)")
        try:
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=object_pos,
                timeout=10.0
            )
            robot_pos = ready_pose
        except Exception as e:
            print(f"[Error] Failed to go to ready pose: {e}")
            return

        # === Step 4: Move to drawer location ===
        print(f"[Task] Moving to drawer '{drawer_name}' at {drawer_pos}")
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_pos,
                to_location=drawer_pos,
                timeout=10.0
            )
            robot_pos = drawer_pos
        except Exception as e:
            print(f"[Error] Failed to move to drawer: {e}")
            return

        # === Step 5: Place object in drawer (sets need-ready) ===
        print(f"[Task] Placing object '{object_name}' into drawer '{drawer_name}'")
        try:
            obs, reward, done = execute_place(
                env,
                task,
                object_name=object_name,
                drawer_name=drawer_name,
                location=drawer_pos,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Failed to place object in drawer: {e}")
            return

        # === Step 6: Go to ready pose (clears need-ready) ===
        print(f"[Task] Going to ready pose to clear (need-ready) after placing")
        try:
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=drawer_pos,
                timeout=10.0
            )
            robot_pos = ready_pose
        except Exception as e:
            print(f"[Error] Failed to go to ready pose after placing: {e}")
            return

        # === Step 7: Push the drawer closed ===
        print(f"[Task] Pushing drawer '{drawer_name}' closed")
        try:
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=drawer_name,
                location=drawer_pos,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Failed to push drawer closed: {e}")
            return

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
