# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer missing predicates.

        # For demonstration, we attempt to use the available skills to interact with all objects and drawers.
        # This will help us discover which predicates are required for successful manipulation.

        # Get lists of objects, drawers, and locations from the environment description if available
        # If not available, try to infer from positions or use placeholder names
        objects = []
        drawers = []
        locations = []
        handles = []

        # Try to extract object, drawer, and location names from positions
        for name in positions.keys():
            if 'drawer' in name:
                drawers.append(name)
            elif 'handle' in name:
                handles.append(name)
            elif 'loc' in name or 'room' in name or 'table' in name:
                locations.append(name)
            else:
                objects.append(name)

        # If locations are not found, use a default location
        if not locations:
            locations = ['location_1']

        # If no drawers or handles, skip those actions
        # For each object, try to pick and place in a drawer if possible
        for obj in objects:
            try:
                # Move robot to the object's location if possible
                obj_pos = positions.get(obj, None)
                robot_loc = None
                if obj_pos is not None:
                    # Find the closest location to the object
                    robot_loc = None
                    min_dist = float('inf')
                    for loc in locations:
                        loc_pos = positions.get(loc, None)
                        if loc_pos is not None:
                            dist = np.linalg.norm(np.array(obj_pos) - np.array(loc_pos))
                            if dist < min_dist:
                                min_dist = dist
                                robot_loc = loc
                else:
                    robot_loc = locations[0]

                # Move to the object's location
                # Use execute_go if available
                if 'execute_go' in globals():
                    obs, reward, done = execute_go(env, task, robot_loc, robot_loc)
                # Try to pick the object
                if 'execute_pick' in globals():
                    obs, reward, done = execute_pick(env, task, obj, robot_loc)
                # Try to place the object in the first drawer if available
                if drawers:
                    drawer = drawers[0]
                    obs, reward, done = execute_place(env, task, obj, drawer, robot_loc)
            except Exception as e:
                print(f"[Exploration] Exception during pick/place for {obj}: {e}")

        # For each drawer, try to open and close it
        for drawer in drawers:
            try:
                # Find a handle for the drawer if available
                handle = None
                for h in handles:
                    # Assume handle-of relationship by name
                    if drawer in h:
                        handle = h
                        break
                if not handle and handles:
                    handle = handles[0]
                # Move to the drawer's location
                drawer_pos = positions.get(drawer, None)
                robot_loc = None
                if drawer_pos is not None:
                    min_dist = float('inf')
                    for loc in locations:
                        loc_pos = positions.get(loc, None)
                        if loc_pos is not None:
                            dist = np.linalg.norm(np.array(drawer_pos) - np.array(loc_pos))
                            if dist < min_dist:
                                min_dist = dist
                                robot_loc = loc
                else:
                    robot_loc = locations[0]
                # Pick the handle if possible
                if handle and 'execute_pick' in globals():
                    obs, reward, done = execute_pick(env, task, handle, robot_loc)
                # Pull (open) the drawer
                if handle and 'execute_pull' in globals():
                    obs, reward, done = execute_pull(env, task, drawer, handle, robot_loc)
                # Push (close) the drawer
                if 'execute_push' in globals():
                    obs, reward, done = execute_push(env, task, drawer, robot_loc)
            except Exception as e:
                print(f"[Exploration] Exception during drawer manipulation for {drawer}: {e}")

        # Try to sweep all objects (if skill available)
        for obj in objects:
            try:
                obj_pos = positions.get(obj, None)
                robot_loc = locations[0]
                if obj_pos is not None:
                    min_dist = float('inf')
                    for loc in locations:
                        loc_pos = positions.get(loc, None)
                        if loc_pos is not None:
                            dist = np.linalg.norm(np.array(obj_pos) - np.array(loc_pos))
                            if dist < min_dist:
                                min_dist = dist
                                robot_loc = loc
                if 'execute_sweep' in globals():
                    obs, reward, done = execute_sweep(env, task, obj, robot_loc)
            except Exception as e:
                print(f"[Exploration] Exception during sweep for {obj}: {e}")

        # Try to use gripper (if skill available)
        try:
            if 'execute_gripper' in globals():
                obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] Exception during gripper action: {e}")

        print("[Exploration] Completed exploration phase. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()