# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We use the available skills to explore the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects/drawers and observe the environment's response.

        # Available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # For exploration, we will:
        # 1. Move to each location (execute_go)
        # 2. Try to pick up objects (execute_pick)
        # 3. Try to pull handles (execute_pull)
        # 4. Try to push drawers (execute_push)
        # 5. Observe which actions fail and infer missing predicates

        # For demonstration, we assume positions contains keys for objects, drawers, handles, and locations.
        # Example: positions = {'object_1': {...}, 'drawer_1': {...}, 'handle_1': {...}, 'location_1': {...}, ...}

        # Extract lists of objects, drawers, handles, and locations
        object_names = [name for name in positions if name.startswith('object')]
        drawer_names = [name for name in positions if name.startswith('drawer')]
        handle_names = [name for name in positions if name.startswith('handle')]
        location_names = [name for name in positions if name.startswith('location')]

        # Helper: get current robot location (if available)
        def get_robot_location():
            # This function should return the current robot location name
            # For now, we try to find the location with 'robot-at' predicate in obs
            for loc in location_names:
                if 'robot-at' in obs and loc in obs['robot-at']:
                    return loc
            # Fallback: just pick the first location
            if location_names:
                return location_names[0]
            return None

        # 1. Try moving to each location
        for loc in location_names:
            try:
                current_loc = get_robot_location()
                if current_loc != loc:
                    print(f"[Exploration] Moving robot from {current_loc} to {loc}")
                    obs, reward, done = execute_go(env, task, from_location=current_loc, to_location=loc)
                    if done:
                        print("[Exploration] Task ended during move!")
                        return
            except Exception as e:
                print(f"[Exploration] Exception during execute_go to {loc}: {e}")

        # 2. Try picking up each object on the floor at each location
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting to pick {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, object_name=obj, location_name=loc)
                    if done:
                        print("[Exploration] Task ended during pick!")
                        return
                except Exception as e:
                    print(f"[Exploration] Exception during execute_pick for {obj} at {loc}: {e}")

        # 3. Try pulling each handle at each location (to check for lock-known predicate)
        for handle in handle_names:
            for drawer in drawer_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Attempting to pull handle {handle} of drawer {drawer} at {loc}")
                        obs, reward, done = execute_pull(env, task, drawer_name=drawer, handle_name=handle, location_name=loc)
                        if done:
                            print("[Exploration] Task ended during pull!")
                            return
                    except Exception as e:
                        print(f"[Exploration] Exception during execute_pull for handle {handle} of drawer {drawer} at {loc}: {e}")

        # 4. Try pushing each drawer at each location
        for drawer in drawer_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting to push drawer {drawer} at {loc}")
                    obs, reward, done = execute_push(env, task, drawer_name=drawer, location_name=loc)
                    if done:
                        print("[Exploration] Task ended during push!")
                        return
                except Exception as e:
                    print(f"[Exploration] Exception during execute_push for drawer {drawer} at {loc}: {e}")

        # 5. Try sweeping each object at each location
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting to sweep {obj} at {loc}")
                    obs, reward, done = execute_sweep(env, task, object_name=obj, location_name=loc)
                    if done:
                        print("[Exploration] Task ended during sweep!")
                        return
                except Exception as e:
                    print(f"[Exploration] Exception during execute_sweep for {obj} at {loc}: {e}")

        # 6. Try rotating (if applicable)
        try:
            print(f"[Exploration] Attempting to rotate")
            obs, reward, done = execute_rotate(env, task)
            if done:
                print("[Exploration] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during execute_rotate: {e}")

        # 7. Try gripper action (if applicable)
        try:
            print(f"[Exploration] Attempting to gripper")
            obs, reward, done = execute_gripper(env, task)
            if done:
                print("[Exploration] Task ended during gripper!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during execute_gripper: {e}")

        print("[Exploration] Exploration phase completed. Check logs for failed actions to infer missing predicates.")

        # === End of Exploration Phase ===

        # TODO: After identifying the missing predicate, update the domain/problem and replan as needed.
        # For now, the code only explores and logs which actions fail, helping to identify the missing predicate.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()