# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate (robot-free) ===
        # The feedback indicates that the predicate (robot-free) is missing or not being set as expected.
        # We will use the available skills to explore the environment and check the robot's state.

        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We will attempt to use the skills in a safe exploratory sequence to check if (robot-free) can be achieved or is missing.

        # 1. Try to move the robot using execute_go between two locations (if available)
        location_names = [k for k, v in positions.items() if 'location' in k or 'loc' in k or 'room' in k]
        if len(location_names) < 2:
            # Fallback: try to find any two distinct positions
            location_names = list(positions.keys())
        if len(location_names) >= 2:
            from_loc = location_names[0]
            to_loc = location_names[1]
            print(f"[Exploration] Attempting to move robot from {from_loc} to {to_loc} using execute_go.")
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=from_loc,
                    to_location=to_loc,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print("[Exploration] execute_go completed.")
            except Exception as e:
                print(f"[Exploration] execute_go failed: {e}")

        # 2. Try to pick up an object using execute_pick
        object_names = [k for k in positions.keys() if 'object' in k or 'obj' in k or 'ball' in k or 'handle' in k]
        if object_names:
            obj_name = object_names[0]
            # Try to find a location for the object
            obj_pos = positions[obj_name]
            # Use the first location as the pick location
            pick_location = location_names[0] if location_names else None
            if pick_location:
                print(f"[Exploration] Attempting to pick {obj_name} at {pick_location} using execute_pick.")
                try:
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        object_name=obj_name,
                        location_name=pick_location,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    print("[Exploration] execute_pick completed.")
                except Exception as e:
                    print(f"[Exploration] execute_pick failed: {e}")

        # 3. Try to place the object in a drawer using execute_place
        drawer_names = [k for k in positions.keys() if 'drawer' in k]
        if object_names and drawer_names and location_names:
            obj_name = object_names[0]
            drawer_name = drawer_names[0]
            place_location = location_names[0]
            print(f"[Exploration] Attempting to place {obj_name} in {drawer_name} at {place_location} using execute_place.")
            try:
                obs, reward, done = execute_place(
                    env,
                    task,
                    object_name=obj_name,
                    drawer_name=drawer_name,
                    location_name=place_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print("[Exploration] execute_place completed.")
            except Exception as e:
                print(f"[Exploration] execute_place failed: {e}")

        # 4. Try to open and close a drawer using execute_pull and execute_push
        if drawer_names and object_names and location_names:
            drawer_name = drawer_names[0]
            handle_name = None
            # Try to find a handle for the drawer
            for name in object_names:
                if 'handle' in name:
                    handle_name = name
                    break
            if handle_name is not None:
                pull_location = location_names[0]
                print(f"[Exploration] Attempting to pull {drawer_name} with {handle_name} at {pull_location} using execute_pull.")
                try:
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        drawer_name=drawer_name,
                        handle_name=handle_name,
                        location_name=pull_location,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    print("[Exploration] execute_pull completed.")
                except Exception as e:
                    print(f"[Exploration] execute_pull failed: {e}")

                print(f"[Exploration] Attempting to push {drawer_name} at {pull_location} using execute_push.")
                try:
                    obs, reward, done = execute_push(
                        env,
                        task,
                        drawer_name=drawer_name,
                        location_name=pull_location,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    print("[Exploration] execute_push completed.")
                except Exception as e:
                    print(f"[Exploration] execute_push failed: {e}")

        # 5. Try to use execute_gripper and execute_sweep for completeness
        print("[Exploration] Attempting to use execute_gripper.")
        try:
            obs, reward, done = execute_gripper(
                env,
                task,
                max_steps=50,
                timeout=5.0
            )
            print("[Exploration] execute_gripper completed.")
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Attempting to use execute_sweep (if any object available).")
        if object_names and location_names:
            obj_name = object_names[0]
            sweep_location = location_names[0]
            try:
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    object_name=obj_name,
                    location_name=sweep_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print("[Exploration] execute_sweep completed.")
            except Exception as e:
                print(f"[Exploration] execute_sweep failed: {e}")

        # 6. After each action, check if the robot is in a "free" state (robot-free)
        # This would typically be checked via the environment's state or observation.
        # For demonstration, print the observation after each action.
        print("[Exploration] Final observation after exploration phase:")
        try:
            final_obs = task.get_observation()
            print(final_obs)
        except Exception as e:
            print(f"[Exploration] Could not retrieve final observation: {e}")

        print("[Exploration] Exploration phase complete. If (robot-free) is not observed in the state, it is likely missing.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()