# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (drawer-closed drawer1) is missing or needs to be checked.
        # We will attempt to use available skills to explore the state of drawer1.

        # Assumptions:
        # - There is a drawer named 'drawer1' and possibly a handle object for it.
        # - The robot can move to the drawer's location and attempt to interact with it.
        # - The available skills are: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Step 1: Find the location of drawer1 and its handle (if any)
        drawer_name = 'drawer1'
        handle_name = None
        drawer_location = None

        # Try to find the handle and location for drawer1
        for obj_name, pos in positions.items():
            if 'handle' in obj_name and drawer_name in obj_name:
                handle_name = obj_name
            if drawer_name in obj_name and 'handle' not in obj_name:
                drawer_location = pos

        # Fallback: If drawer_location is not found, use a default or the first location
        if drawer_location is None:
            for obj_name, pos in positions.items():
                if 'drawer' in obj_name and 'handle' not in obj_name:
                    drawer_location = pos
                    break

        # Fallback: If handle_name is not found, try to find any handle
        if handle_name is None:
            for obj_name in positions.keys():
                if 'handle' in obj_name:
                    handle_name = obj_name
                    break

        # Step 2: Move robot to the drawer's location
        # We need to know the robot's current location and the drawer's location name
        # For this example, we assume location names are available as keys in positions
        robot_location = None
        for obj_name in positions.keys():
            if 'robot' in obj_name:
                robot_location = obj_name
                break

        # If robot_location is not found, use a default
        if robot_location is None:
            robot_location = 'robot_base'

        # If drawer_location is a position, we need a location name
        # For this example, we assume the location name is 'drawer1_location'
        drawer_location_name = None
        for loc_name in positions.keys():
            if drawer_name in loc_name and 'location' in loc_name:
                drawer_location_name = loc_name
                break
        if drawer_location_name is None:
            drawer_location_name = 'drawer1_location'

        # Step 3: Use execute_go to move to the drawer's location
        try:
            print(f"[Exploration] Moving robot from {robot_location} to {drawer_location_name}")
            obs, reward, done = execute_go(env, task, robot_location, drawer_location_name)
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")

        # Step 4: Try to pull the drawer to check its state (drawer-closed, drawer-open, etc.)
        # This will help us determine if the predicate (drawer-closed drawer1) is missing or not.
        # We need to pick the handle first if required.

        # Try to pick the handle (if available)
        if handle_name is not None:
            try:
                print(f"[Exploration] Attempting to pick handle {handle_name} at {drawer_location_name}")
                obs, reward, done = execute_pick(env, task, handle_name, drawer_location_name)
            except Exception as e:
                print(f"[Exploration] Error during execute_pick (handle): {e}")

            # Try to pull the drawer using the handle
            try:
                print(f"[Exploration] Attempting to pull drawer {drawer_name} using handle {handle_name} at {drawer_location_name}")
                obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_location_name)
            except Exception as e:
                print(f"[Exploration] Error during execute_pull: {e}")
        else:
            print("[Exploration] No handle found for drawer; skipping pick and pull.")

        # Step 5: Try to push the drawer (to close it) to further explore its state
        try:
            print(f"[Exploration] Attempting to push (close) drawer {drawer_name} at {drawer_location_name}")
            obs, reward, done = execute_push(env, task, drawer_name, drawer_location_name)
        except Exception as e:
            print(f"[Exploration] Error during execute_push: {e}")

        # Step 6: Exploration complete. At this point, the robot has attempted to interact with the drawer.
        # The results (success/failure) and the environment's feedback will help identify if the predicate (drawer-closed drawer1) is missing.

        print("[Exploration] Exploration phase complete. Check logs or environment state for missing predicate information.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()