# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and try to discover missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # We will attempt to use the available skills to interact with objects and drawers,
        # and observe if any state changes or errors occur, which may indicate missing predicates.

        # Get all objects, drawers, and locations from positions
        objects = []
        drawers = []
        locations = []
        for name, pos in positions.items():
            if 'drawer' in name:
                drawers.append(name)
            elif 'loc' in name or 'location' in name or 'room' in name:
                locations.append(name)
            else:
                objects.append(name)

        # If locations are not explicitly given, infer from object positions
        if not locations:
            # Assume all unique (x, y) positions are locations
            loc_set = set()
            for pos in positions.values():
                loc_set.add(tuple(pos[:2]))
            locations = ['loc_{}'.format(i) for i in range(len(loc_set))]

        # Try to execute each available skill in a safe way to probe for missing predicates
        # We focus on skills that may reveal missing predicates: execute_pick, execute_pull, execute_place, execute_push, execute_go, execute_sweep, execute_gripper

        # 1. Try to pick up each object from the floor at its location
        for obj in objects:
            try:
                obj_pos = positions[obj]
                # Find the nearest location for the object
                # (Assume location names are not available, so use object's own position)
                print(f"[Exploration] Attempting to pick up object: {obj}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target_obj=obj,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Successfully picked up {obj}.")
                    # Try to place it in a drawer if possible
                    for drawer in drawers:
                        try:
                            drawer_pos = positions[drawer]
                            print(f"[Exploration] Attempting to place {obj} in {drawer}")
                            obs, reward, done = execute_place(
                                env,
                                task,
                                target_obj=obj,
                                target_drawer=drawer,
                                target_pos=drawer_pos,
                                approach_distance=0.15,
                                max_steps=100,
                                threshold=0.01,
                                approach_axis='z',
                                timeout=10.0
                            )
                            if done:
                                print(f"[Exploration] Successfully placed {obj} in {drawer}.")
                                break
                        except Exception as e:
                            print(f"[Exploration] Could not place {obj} in {drawer}: {e}")
                    break
            except Exception as e:
                print(f"[Exploration] Could not pick up {obj}: {e}")

        # 2. Try to open and close each drawer using pull and push
        for drawer in drawers:
            # Try to find a handle for the drawer
            handle_name = None
            for obj in objects:
                if 'handle' in obj and drawer in obj:
                    handle_name = obj
                    break
            if not handle_name:
                # Try to guess handle name
                for obj in objects:
                    if 'handle' in obj:
                        handle_name = obj
                        break
            if not handle_name:
                print(f"[Exploration] No handle found for {drawer}, skipping pull.")
                continue
            try:
                handle_pos = positions[handle_name]
                drawer_pos = positions[drawer]
                print(f"[Exploration] Attempting to pick up handle {handle_name} for {drawer}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target_obj=handle_name,
                    target_pos=handle_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Picked up handle {handle_name}, now pulling {drawer}")
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        target_drawer=drawer,
                        target_handle=handle_name,
                        target_pos=drawer_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Successfully pulled (opened) {drawer}.")
                        # Try to push (close) the drawer
                        print(f"[Exploration] Attempting to push (close) {drawer}")
                        obs, reward, done = execute_push(
                            env,
                            task,
                            target_drawer=drawer,
                            target_pos=drawer_pos,
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        if done:
                            print(f"[Exploration] Successfully pushed (closed) {drawer}.")
            except Exception as e:
                print(f"[Exploration] Could not pull/push {drawer}: {e}")

        # 3. Try to move (go) between locations if possible
        if len(locations) >= 2:
            from_loc = locations[0]
            to_loc = locations[1]
            print(f"[Exploration] Attempting to move from {from_loc} to {to_loc}")
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=from_loc,
                    to_location=to_loc,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}")
            except Exception as e:
                print(f"[Exploration] Could not move from {from_loc} to {to_loc}: {e}")

        # 4. Try to sweep each object (if available)
        for obj in objects:
            try:
                obj_pos = positions[obj]
                print(f"[Exploration] Attempting to sweep object: {obj}")
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    target_obj=obj,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Successfully swept {obj}.")
            except Exception as e:
                print(f"[Exploration] Could not sweep {obj}: {e}")

        # 5. Try to use gripper (if available)
        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(
                env,
                task,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print(f"[Exploration] Successfully used gripper.")
        except Exception as e:
            print(f"[Exploration] Could not use gripper: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()