# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Exploration Phase: Identify Missing Predicate ---
        # Feedback: (robot-at drawer-area)
        # The missing predicate is likely related to the robot's location, specifically (robot-at drawer-area).
        # We will use available skills to explore and confirm the robot's location predicates.

        # Step 1: Try to move the robot to the 'drawer-area' (if such a location exists)
        # Step 2: After moving, check if the robot is at the expected location (drawer-area)
        # Step 3: If not, try to explore other locations or log the missing predicate

        # For demonstration, we assume 'drawer-area' is a key in positions
        # and that the robot starts at some initial location.

        # Find all locations from positions
        location_keys = [k for k, v in positions.items() if 'area' in k or 'location' in k or 'drawer' in k]
        # Fallback: if no such keys, just use all keys
        if not location_keys:
            location_keys = list(positions.keys())

        # Try to find 'drawer-area'
        drawer_area = None
        for k in location_keys:
            if 'drawer' in k and 'area' in k:
                drawer_area = k
                break
        if drawer_area is None:
            # Try fallback: any 'drawer' location
            for k in location_keys:
                if 'drawer' in k:
                    drawer_area = k
                    break

        # Get robot's current location (assume it's in descriptions or obs)
        # This is highly environment-specific; fallback to first location
        robot_current_location = None
        for k in location_keys:
            if 'robot' in k:
                robot_current_location = k
                break
        if robot_current_location is None and location_keys:
            robot_current_location = location_keys[0]

        print(f"[Exploration] Robot current location: {robot_current_location}")
        print(f"[Exploration] Target drawer-area: {drawer_area}")

        # If both locations are found, try to move the robot
        if robot_current_location and drawer_area and robot_current_location != drawer_area:
            try:
                print(f"[Exploration] Moving robot from {robot_current_location} to {drawer_area} using execute_go.")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_current_location,
                    to_location=drawer_area,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] Robot moved to {drawer_area}.")
            except Exception as e:
                print(f"[Exploration] Error during execute_go: {e}")
        else:
            print("[Exploration] Could not determine robot or drawer-area location for exploration.")

        # After moving, check if the predicate (robot-at drawer-area) is now true
        # This would be part of the observation or environment state
        # For demonstration, print out the current positions again
        positions_after = get_object_positions()
        print(f"[Exploration] Positions after move: {positions_after}")

        # If the predicate is still missing, log it
        if drawer_area not in positions_after.get('robot', ''):
            print("[Exploration] Predicate (robot-at drawer-area) may be missing or not updated in environment state.")

        # --- End of Exploration Phase ---

        # === (Optional) Continue with Oracle Plan Execution ===
        # Here you would continue with the oracle plan, using only the available skills.
        # For this exploration task, we focus on identifying the missing predicate.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()