# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate (robot-at drawer-area) ===
        # The feedback indicates that the predicate (robot-at drawer-area) is missing.
        # We need to explore the environment to determine if the robot can be at 'drawer-area'.
        # We'll use the available skill 'execute_go' to move the robot to 'drawer-area' and check the result.

        # Assume 'drawer-area' is a key in positions, or fallback to a default location name
        drawer_area = None
        for key in positions:
            if 'drawer' in key and 'area' in key:
                drawer_area = key
                break
        if drawer_area is None:
            # Fallback: try common names
            if 'drawer-area' in positions:
                drawer_area = 'drawer-area'
            elif 'drawer' in positions:
                drawer_area = 'drawer'
            else:
                # If not found, pick any location as a placeholder
                drawer_area = list(positions.keys())[0]

        # Find current robot location
        robot_location = None
        for key in positions:
            if 'robot' in key or 'base' in key:
                robot_location = key
                break
        if robot_location is None:
            # Fallback: try to find a location that is not an object
            robot_location = list(positions.keys())[0]

        print(f"[Exploration] Moving robot from {robot_location} to {drawer_area} to check predicate (robot-at drawer-area)")

        try:
            # Use the predefined skill 'execute_go' to move the robot
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=drawer_area,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] Robot moved to {drawer_area}. Checking if predicate (robot-at {drawer_area}) holds.")
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")
            return

        # Optionally, check the observation or state to confirm the predicate
        # (Assume task.get_observation() returns a dict or similar structure)
        try:
            current_obs = task.get_observation()
            if hasattr(current_obs, 'get'):
                robot_at_drawer = current_obs.get('robot-at', None)
                if robot_at_drawer == drawer_area:
                    print(f"[Exploration] Predicate (robot-at {drawer_area}) is now true.")
                else:
                    print(f"[Exploration] Predicate (robot-at {drawer_area}) is still missing or not detected.")
            else:
                print("[Exploration] Unable to check predicate in observation format.")
        except Exception as e:
            print(f"[Exploration] Error checking observation: {e}")

        # === Continue with Oracle Plan (if any) ===
        # Here you would continue with the rest of the oracle plan, using only the available skills.
        # For example, you might pick, place, pull, or push objects as required by the plan.
        # Since the main focus is on exploration for the missing predicate, we stop here.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()