# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer missing predicates.

        # For demonstration, we attempt to use all available skills in a safe exploratory sequence.
        # This will help us discover which predicates are required for successful planning.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Attempt to explore the environment using available skills
        # We assume the environment provides at least one object, one drawer, and locations
        # For robustness, we check for their existence

        # Extract object, drawer, and location names from positions (if available)
        object_names = []
        drawer_names = []
        location_names = []

        for name in positions.keys():
            if 'drawer' in name:
                drawer_names.append(name)
            elif 'loc' in name or 'location' in name or 'room' in name:
                location_names.append(name)
            else:
                object_names.append(name)

        # Fallback: If no names found, use generic placeholders
        if not object_names:
            object_names = ['object_1']
        if not drawer_names:
            drawer_names = ['drawer_1']
        if not location_names:
            location_names = ['location_1', 'location_2']

        # Select first available object, drawer, and locations for exploration
        obj = object_names[0]
        drawer = drawer_names[0]
        loc1 = location_names[0]
        loc2 = location_names[1] if len(location_names) > 1 else location_names[0]

        # 1. Try moving the robot between locations
        try:
            print(f"[Exploration] Moving robot from {loc1} to {loc2} using execute_go.")
            obs, reward, done = execute_go(env, task, loc1, loc2)
        except Exception as e:
            print(f"[Exploration] execute_go failed: {e}")

        # 2. Try picking up an object
        try:
            print(f"[Exploration] Attempting to pick up {obj} at {loc2} using execute_pick.")
            obs, reward, done = execute_pick(env, task, obj, loc2)
        except Exception as e:
            print(f"[Exploration] execute_pick failed: {e}")

        # 3. Try pulling the drawer handle (if handle exists)
        # We assume the handle is named as 'handle_of_' + drawer
        handle_name = f"handle_of_{drawer}"
        if handle_name in positions:
            handle = handle_name
        else:
            # Fallback: use the object as handle if no handle found
            handle = obj

        try:
            print(f"[Exploration] Attempting to pull {drawer} using handle {handle} at {loc2} with execute_pull.")
            obs, reward, done = execute_pull(env, task, drawer, handle, loc2)
        except Exception as e:
            print(f"[Exploration] execute_pull failed: {e}")

        # 4. Try placing the object in the drawer
        try:
            print(f"[Exploration] Attempting to place {obj} in {drawer} at {loc2} using execute_place.")
            obs, reward, done = execute_place(env, task, obj, drawer, loc2)
        except Exception as e:
            print(f"[Exploration] execute_place failed: {e}")

        # 5. Try pushing the drawer closed
        try:
            print(f"[Exploration] Attempting to push {drawer} closed at {loc2} using execute_push.")
            obs, reward, done = execute_push(env, task, drawer, loc2)
        except Exception as e:
            print(f"[Exploration] execute_push failed: {e}")

        # 6. Try sweeping the object
        try:
            print(f"[Exploration] Attempting to sweep {obj} at {loc2} using execute_sweep.")
            obs, reward, done = execute_sweep(env, task, obj, loc2)
        except Exception as e:
            print(f"[Exploration] execute_sweep failed: {e}")

        # 7. Try rotating (if available)
        try:
            print(f"[Exploration] Attempting to rotate using execute_rotate.")
            obs, reward, done = execute_rotate(env, task)
        except Exception as e:
            print(f"[Exploration] execute_rotate failed: {e}")

        # 8. Try gripper action (if available)
        try:
            print(f"[Exploration] Attempting gripper action using execute_gripper.")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()