# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # For demonstration, we will attempt to use the available skills to interact with objects and drawers,
        # and print out the results to help identify which predicate might be missing.

        # Get all objects and locations from positions
        object_names = [name for name in positions if positions[name].get('type', '') == 'object']
        drawer_names = [name for name in positions if positions[name].get('type', '') == 'drawer']
        location_names = [name for name in positions if positions[name].get('type', '') == 'location']

        # If types are not provided, fallback to all keys
        if not object_names:
            object_names = [name for name in positions if 'object' in name]
        if not drawer_names:
            drawer_names = [name for name in positions if 'drawer' in name]
        if not location_names:
            location_names = [name for name in positions if 'location' in name or 'room' in name or 'table' in name]

        # Try to find the robot's initial location
        robot_location = None
        for name in location_names:
            if positions[name].get('robot_here', False):
                robot_location = name
                break
        if robot_location is None and location_names:
            robot_location = location_names[0]

        print("[Exploration] Robot initial location:", robot_location)
        print("[Exploration] Objects:", object_names)
        print("[Exploration] Drawers:", drawer_names)
        print("[Exploration] Locations:", location_names)

        # Try to move to each location and interact with objects/drawers
        for loc in location_names:
            if loc != robot_location:
                try:
                    print(f"[Exploration] Moving robot from {robot_location} to {loc}")
                    obs, reward, done = execute_go(env, task, robot_location, loc)
                    robot_location = loc
                except Exception as e:
                    print(f"[Exploration] Failed to move to {loc}: {e}")

            # Try to pick up any object at this location
            for obj in object_names:
                try:
                    print(f"[Exploration] Attempting to pick {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    print(f"[Exploration] Picked {obj} at {loc}")
                    # Try to place in any drawer at this location
                    for drawer in drawer_names:
                        try:
                            print(f"[Exploration] Attempting to place {obj} in {drawer} at {loc}")
                            obs, reward, done = execute_place(env, task, obj, drawer, loc)
                            print(f"[Exploration] Placed {obj} in {drawer} at {loc}")
                        except Exception as e:
                            print(f"[Exploration] Failed to place {obj} in {drawer} at {loc}: {e}")
                    # Try to drop the object if not placed
                    try:
                        print(f"[Exploration] Attempting to sweep {obj} at {loc}")
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                    except Exception as e:
                        print(f"[Exploration] Failed to sweep {obj} at {loc}: {e}")
                except Exception as e:
                    print(f"[Exploration] Failed to pick {obj} at {loc}: {e}")

            # Try to interact with drawers: pull (open) and push (close)
            for drawer in drawer_names:
                # Try to find a handle for the drawer
                handle_name = None
                for obj in object_names:
                    if 'handle' in obj and obj in positions and positions[obj].get('drawer', '') == drawer:
                        handle_name = obj
                        break
                if handle_name:
                    try:
                        print(f"[Exploration] Attempting to pick handle {handle_name} at {loc}")
                        obs, reward, done = execute_pick(env, task, handle_name, loc)
                        print(f"[Exploration] Picked handle {handle_name} at {loc}")
                        try:
                            print(f"[Exploration] Attempting to pull (open) drawer {drawer} with handle {handle_name} at {loc}")
                            obs, reward, done = execute_pull(env, task, drawer, handle_name, loc)
                            print(f"[Exploration] Pulled (opened) drawer {drawer} at {loc}")
                        except Exception as e:
                            print(f"[Exploration] Failed to pull (open) drawer {drawer} at {loc}: {e}")
                    except Exception as e:
                        print(f"[Exploration] Failed to pick handle {handle_name} at {loc}: {e}")
                # Try to push (close) the drawer
                try:
                    print(f"[Exploration] Attempting to push (close) drawer {drawer} at {loc}")
                    obs, reward, done = execute_push(env, task, drawer, loc)
                    print(f"[Exploration] Pushed (closed) drawer {drawer} at {loc}")
                except Exception as e:
                    print(f"[Exploration] Failed to push (close) drawer {drawer} at {loc}: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()