# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at drawer-area)
        # The predicate (robot-at drawer-area) is missing or not being set as expected.
        # We need to explore the environment to determine if the robot can be at 'drawer-area'
        # and which skill/action is responsible for setting this predicate.

        # We will use the available skills to move the robot to 'drawer-area' and check the result.
        # The relevant skill is likely 'execute_go' (move robot between locations).

        # For demonstration, let's assume the robot starts at 'start-area' and we want to move to 'drawer-area'.
        # We'll use the positions dictionary to get the actual location names if available.

        # Find possible locations
        location_names = []
        for name, pos in positions.items():
            if 'area' in name or 'location' in name:
                location_names.append(name)
        # Fallback if not found
        if not location_names:
            location_names = ['start-area', 'drawer-area']

        # Try to determine from observation or positions where the robot is
        current_location = None
        for name in location_names:
            if 'start' in name:
                current_location = name
                break
        if current_location is None:
            current_location = location_names[0]
        target_location = None
        for name in location_names:
            if 'drawer' in name:
                target_location = name
                break
        if target_location is None:
            target_location = location_names[-1]

        print(f"[Exploration] Current location: {current_location}")
        print(f"[Exploration] Target location: {target_location}")

        # Use execute_go to move the robot to the drawer-area
        try:
            print(f"[Exploration] Executing execute_go from {current_location} to {target_location}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=current_location,
                to_location=target_location
            )
            print(f"[Exploration] Robot moved to {target_location}.")
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")

        # After moving, check if the predicate (robot-at drawer-area) is now true in the environment.
        # This would typically be done by querying the environment state or observation.
        # For demonstration, print the observation or relevant info.
        try:
            obs = task.get_observation()
            print(f"[Exploration] Observation after move: {obs}")
            # If obs is a dict or has a state, try to print robot location
            if isinstance(obs, dict):
                robot_loc = obs.get('robot_location', None)
                print(f"[Exploration] Robot location in observation: {robot_loc}")
        except Exception as e:
            print(f"[Exploration] Error retrieving observation: {e}")

        # If the predicate is still missing, further exploration can be done using other skills,
        # such as execute_go_identify, execute_go_temperature, etc., if available.
        # However, per the available skills, only execute_go is relevant for moving.

        # === End of Exploration Phase ===

        # TODO: Continue with the main oracle plan after confirming the missing predicate is handled.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
