# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (drawer-closed drawer1) is missing or needs to be explored.
        # We will use available skills to explore the state of drawer1.

        # Assumptions:
        # - There is a drawer named 'drawer1' and possibly a handle object for it.
        # - The robot can move to the drawer's location and attempt to interact with it.
        # - The available skills are: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Step 1: Get relevant object/location names
        # Try to extract drawer, handle, and location names from positions or descriptions
        drawer_name = None
        handle_name = None
        drawer_location = None
        robot_location = None

        # Try to infer names from positions or descriptions
        for name in positions:
            if 'drawer' in name:
                drawer_name = name
            if 'handle' in name:
                handle_name = name
            if 'loc' in name or 'table' in name or 'room' in name:
                drawer_location = name
            if 'robot' in name or 'start' in name:
                robot_location = name

        # Fallbacks if not found
        if drawer_name is None:
            drawer_name = 'drawer1'
        if handle_name is None:
            handle_name = 'handle1'
        if drawer_location is None:
            drawer_location = 'location1'
        if robot_location is None:
            robot_location = 'location0'

        print(f"[Exploration] Using drawer: {drawer_name}, handle: {handle_name}, drawer_location: {drawer_location}, robot_location: {robot_location}")

        # Step 2: Move robot to drawer location if not already there
        try:
            # If robot is not at drawer_location, move there
            if robot_location != drawer_location:
                print(f"[Exploration] Moving robot from {robot_location} to {drawer_location}")
                obs, reward, done = execute_go(env, task, robot_location, drawer_location)
                robot_location = drawer_location
        except Exception as e:
            print(f"[Exploration] Exception during execute_go: {e}")

        # Step 3: Try to pick the handle (if available) to prepare for pull
        try:
            print(f"[Exploration] Attempting to pick handle: {handle_name} at {drawer_location}")
            obs, reward, done = execute_pick(env, task, handle_name, drawer_location)
        except Exception as e:
            print(f"[Exploration] Exception during execute_pick: {e}")

        # Step 4: Try to pull the drawer to check its state (this may reveal lock/closed predicates)
        try:
            print(f"[Exploration] Attempting to pull drawer: {drawer_name} with handle: {handle_name} at {drawer_location}")
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_location)
        except Exception as e:
            print(f"[Exploration] Exception during execute_pull: {e}")

        # Step 5: Try to push the drawer (close it) to see if the predicate (drawer-closed drawer1) can be established
        try:
            print(f"[Exploration] Attempting to push (close) drawer: {drawer_name} at {drawer_location}")
            obs, reward, done = execute_push(env, task, drawer_name, drawer_location)
        except Exception as e:
            print(f"[Exploration] Exception during execute_push: {e}")

        # Step 6: (Optional) Try to place an object in the drawer if needed for further exploration
        # This step is not strictly necessary for (drawer-closed drawer1), but can be added if required

        print("[Exploration] Exploration phase complete. Check logs or state for (drawer-closed drawer1) predicate.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()