# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate ===
        # Feedback: (on-floor dice2)
        # The goal is to explore the environment and determine which predicate is missing
        # by using only the available skills.

        # Available skills:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We will attempt to use the available skills to interact with the object 'dice2'
        # and see if any action fails due to a missing predicate, or if we can infer the missing predicate.

        # For demonstration, let's assume 'dice2' is present in positions and is on the floor.
        # We'll try to pick it up using 'execute_pick' and handle any errors.

        # Find the location of dice2
        dice2_location = None
        for obj_name, pos in positions.items():
            if 'dice2' in obj_name:
                dice2_location = pos
                break

        if dice2_location is None:
            print("[Exploration] dice2 not found in object positions. Exploration aborted.")
        else:
            print(f"[Exploration] dice2 found at position: {dice2_location}")

            # Try to pick up dice2 using execute_pick
            try:
                print("[Exploration] Attempting to pick up dice2 using execute_pick...")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target_obj='dice2',
                    target_pos=dice2_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Successfully picked up dice2. Predicate (on-floor dice2) is valid and handled by execute_pick.")
                else:
                    print("[Exploration] Could not pick up dice2. There may be a missing predicate or precondition.")
            except Exception as e:
                print(f"[Exploration] Exception occurred during execute_pick: {e}")
                print("[Exploration] This may indicate a missing predicate or an unmet precondition for picking up dice2.")

            # If picking up fails, try to sweep as an alternative
            if not done:
                try:
                    print("[Exploration] Attempting to sweep dice2 using execute_sweep...")
                    obs, reward, done = execute_sweep(
                        env,
                        task,
                        target_obj='dice2',
                        target_pos=dice2_location,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print("[Exploration] Successfully swept dice2. This may indicate that (on-floor dice2) is only actionable via sweep.")
                    else:
                        print("[Exploration] Could not sweep dice2. Further exploration needed.")
                except Exception as e:
                    print(f"[Exploration] Exception occurred during execute_sweep: {e}")

        # === End of Exploration Phase ===

        # At this point, based on the feedback and exploration, we can infer that the missing predicate is likely (on-floor dice2)
        # and that the action execute_pick requires this predicate to be true for dice2.

        # If further exploration is needed, repeat similar logic for other objects or actions.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()