# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Discover Missing Predicate ===
        # Feedback: (on-floor drawer2)
        # The goal is to explore the environment to determine if 'on-floor' applies to drawer2,
        # and to check if any predicate is missing for correct planning/execution.

        # We will use only the available skills:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # For exploration, we will attempt to interact with 'drawer2' as if it is an object on the floor.
        # If the action fails, we infer that the predicate or type is missing or incorrect.

        # Step 1: Find the location of 'drawer2' (if available)
        drawer2_pos = None
        for name in positions:
            if 'drawer2' in name:
                drawer2_pos = positions[name]
                break

        if drawer2_pos is None:
            print("[Exploration] Could not find position for 'drawer2'. Exploration aborted.")
        else:
            print(f"[Exploration] Found 'drawer2' at position: {drawer2_pos}")

            # Step 2: Move robot to the location of 'drawer2'
            # We need to know the robot's current location and the location name for 'drawer2'
            # For this example, we assume the location names are available as keys in positions
            # and that the robot's current location is also available.

            # Find robot's current location
            robot_loc = None
            for name in positions:
                if 'robot' in name:
                    robot_loc = positions[name]
                    break

            # For demonstration, we use the location name as the key for execute_go
            # If not available, we skip the move step
            try:
                if robot_loc is not None:
                    print(f"[Exploration] Moving robot from {robot_loc} to {drawer2_pos}")
                    obs, reward, done = execute_go(env, task, robot_loc, drawer2_pos)
                else:
                    print("[Exploration] Robot location unknown, skipping move.")
            except Exception as e:
                print(f"[Exploration] Exception during execute_go: {e}")

            # Step 3: Try to pick 'drawer2' as if it is an object on the floor
            # This will test if the predicate (on-floor drawer2) is valid and if the action is possible
            try:
                print("[Exploration] Attempting to pick 'drawer2' using execute_pick...")
                obs, reward, done = execute_pick(env, task, 'drawer2', drawer2_pos)
                print("[Exploration] execute_pick on 'drawer2' succeeded. 'on-floor drawer2' is a valid predicate.")
            except Exception as e:
                print(f"[Exploration] execute_pick on 'drawer2' failed: {e}")
                print("[Exploration] This suggests a missing or incorrect predicate/type for 'drawer2'.")

            # Step 4: Try to sweep 'drawer2' (if picking is not possible, maybe sweeping is)
            try:
                print("[Exploration] Attempting to sweep 'drawer2' using execute_sweep...")
                obs, reward, done = execute_sweep(env, task, 'drawer2', drawer2_pos)
                print("[Exploration] execute_sweep on 'drawer2' succeeded.")
            except Exception as e:
                print(f"[Exploration] execute_sweep on 'drawer2' failed: {e}")

            # Step 5: Try to pull 'drawer2' (if it has a handle and is a drawer)
            try:
                print("[Exploration] Attempting to pull 'drawer2' using execute_pull...")
                # For execute_pull, we need the drawer, handle, and location
                # We'll try with 'drawer2' as both the drawer and the handle for exploration
                obs, reward, done = execute_pull(env, task, 'drawer2', 'drawer2', drawer2_pos)
                print("[Exploration] execute_pull on 'drawer2' succeeded.")
            except Exception as e:
                print(f"[Exploration] execute_pull on 'drawer2' failed: {e}")

        # === End of Exploration Phase ===
        print("[Exploration] Exploration phase complete. Check above for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
