# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate (robot-free) ===
        # The feedback indicates that the predicate (robot-free) is missing or not being set as expected.
        # We will use the available skills to explore the environment and check the robot's state after each action.

        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We will attempt to execute each skill in a safe, exploratory manner and observe the robot's state.
        # This will help us determine when (robot-free) becomes true or false.

        # For demonstration, we will:
        # 1. Move the robot to a location (execute_go)
        # 2. Attempt to pick an object (execute_pick)
        # 3. Place the object (execute_place) if possible
        # 4. Try to open/close a drawer (execute_pull, execute_push)
        # 5. After each action, check if the robot is "free" (i.e., not holding anything, hand is empty, etc.)

        # --- Step 1: Identify objects, drawers, locations from positions dict ---
        # This is a generic approach; adapt keys as needed for your environment
        objects = []
        drawers = []
        locations = []
        handles = []
        for name, pos in positions.items():
            if 'drawer' in name:
                drawers.append(name)
            elif 'handle' in name:
                handles.append(name)
            elif 'loc' in name or 'location' in name or 'room' in name:
                locations.append(name)
            else:
                objects.append(name)

        # Fallback: If no explicit locations, use unique positions as locations
        if not locations:
            unique_locs = set()
            for pos in positions.values():
                unique_locs.add(tuple(np.round(pos, 2)))
            locations = [f"loc_{i}" for i in range(len(unique_locs))]

        # Pick a default object, drawer, handle, and location for exploration
        obj = objects[0] if objects else None
        drawer = drawers[0] if drawers else None
        handle = handles[0] if handles else None
        location = locations[0] if locations else None
        location2 = locations[1] if len(locations) > 1 else location

        # --- Step 2: Exploration using available skills ---
        # We will try to execute each skill and print the robot's state after each

        print("[Exploration] Starting exploration to find missing predicate (robot-free)")

        # Helper: Check robot-free predicate (if available in observation)
        def check_robot_free(obs):
            # Try to infer from obs dict if robot is free
            # This is a placeholder; adapt to your observation structure
            if isinstance(obs, dict):
                if 'robot-free' in obs:
                    return obs['robot-free']
                if 'hand-empty' in obs:
                    return obs['hand-empty']
                if 'holding' in obs:
                    return not obs['holding']
            return None

        # 1. Move robot to another location
        try:
            print(f"[Exploration] execute_go: {location} -> {location2}")
            obs, reward, done, info = execute_go(env, task, from_location=location, to_location=location2)
            robot_free = check_robot_free(obs)
            print(f"After execute_go: robot-free = {robot_free}")
        except Exception as e:
            print(f"execute_go failed: {e}")

        # 2. Try to pick an object
        try:
            print(f"[Exploration] execute_pick: {obj} at {location2}")
            obs, reward, done, info = execute_pick(env, task, object_name=obj, location=location2)
            robot_free = check_robot_free(obs)
            print(f"After execute_pick: robot-free = {robot_free}")
        except Exception as e:
            print(f"execute_pick failed: {e}")

        # 3. Try to place the object in a drawer
        try:
            if drawer:
                print(f"[Exploration] execute_place: {obj} into {drawer} at {location2}")
                obs, reward, done, info = execute_place(env, task, object_name=obj, drawer_name=drawer, location=location2)
                robot_free = check_robot_free(obs)
                print(f"After execute_place: robot-free = {robot_free}")
        except Exception as e:
            print(f"execute_place failed: {e}")

        # 4. Try to open the drawer (pull)
        try:
            if drawer and handle:
                print(f"[Exploration] execute_pull: {drawer} with {handle} at {location2}")
                obs, reward, done, info = execute_pull(env, task, drawer_name=drawer, handle_name=handle, location=location2)
                robot_free = check_robot_free(obs)
                print(f"After execute_pull: robot-free = {robot_free}")
        except Exception as e:
            print(f"execute_pull failed: {e}")

        # 5. Try to close the drawer (push)
        try:
            if drawer:
                print(f"[Exploration] execute_push: {drawer} at {location2}")
                obs, reward, done, info = execute_push(env, task, drawer_name=drawer, location=location2)
                robot_free = check_robot_free(obs)
                print(f"After execute_push: robot-free = {robot_free}")
        except Exception as e:
            print(f"execute_push failed: {e}")

        # 6. Try to sweep an object
        try:
            print(f"[Exploration] execute_sweep: {obj} at {location2}")
            obs, reward, done, info = execute_sweep(env, task, object_name=obj, location=location2)
            robot_free = check_robot_free(obs)
            print(f"After execute_sweep: robot-free = {robot_free}")
        except Exception as e:
            print(f"execute_sweep failed: {e}")

        # 7. Try to use gripper
        try:
            print(f"[Exploration] execute_gripper")
            obs, reward, done, info = execute_gripper(env, task)
            robot_free = check_robot_free(obs)
            print(f"After execute_gripper: robot-free = {robot_free}")
        except Exception as e:
            print(f"execute_gripper failed: {e}")

        print("[Exploration] Exploration complete. Check above outputs for robot-free predicate transitions.")

        # === End of Exploration Phase ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()