# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available exploration skills to probe the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to check for missing predicates.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For demonstration, we will attempt to use each skill in a safe, exploratory way.
        # This will help us determine if any predicate is missing or if an action fails due to a missing precondition.

        # Get all objects, drawers, and locations from positions
        object_names = [name for name in positions if 'object' in name]
        drawer_names = [name for name in positions if 'drawer' in name]
        location_names = [name for name in positions if 'location' in name or 'room' in name or 'table' in name]

        # For exploration, pick the first available object, drawer, and location if present
        test_object = object_names[0] if object_names else None
        test_drawer = drawer_names[0] if drawer_names else None
        test_location = location_names[0] if location_names else None

        # Try to execute each skill and catch exceptions to infer missing predicates
        print("[Exploration] Starting exploration to identify missing predicates...")
        exploration_results = {}

        # 1. Try execute_pick
        if 'execute_pick' in available_skills and test_object and test_location:
            try:
                print(f"[Exploration] Trying execute_pick on {test_object} at {test_location}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj_name=test_object,
                    location_name=test_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                exploration_results['execute_pick'] = "Success"
            except Exception as e:
                print(f"[Exploration] execute_pick failed: {e}")
                exploration_results['execute_pick'] = f"Failed: {e}"

        # 2. Try execute_place
        if 'execute_place' in available_skills and test_object and test_drawer and test_location:
            try:
                print(f"[Exploration] Trying execute_place for {test_object} into {test_drawer} at {test_location}")
                obs, reward, done = execute_place(
                    env,
                    task,
                    obj_name=test_object,
                    drawer_name=test_drawer,
                    location_name=test_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                exploration_results['execute_place'] = "Success"
            except Exception as e:
                print(f"[Exploration] execute_place failed: {e}")
                exploration_results['execute_place'] = f"Failed: {e}"

        # 3. Try execute_push (close drawer)
        if 'execute_push' in available_skills and test_drawer and test_location:
            try:
                print(f"[Exploration] Trying execute_push on {test_drawer} at {test_location}")
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer_name=test_drawer,
                    location_name=test_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                exploration_results['execute_push'] = "Success"
            except Exception as e:
                print(f"[Exploration] execute_push failed: {e}")
                exploration_results['execute_push'] = f"Failed: {e}"

        # 4. Try execute_pull (open drawer)
        if 'execute_pull' in available_skills and test_drawer and test_object and test_location:
            try:
                print(f"[Exploration] Trying execute_pull on {test_drawer} with handle {test_object} at {test_location}")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer_name=test_drawer,
                    handle_name=test_object,
                    location_name=test_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                exploration_results['execute_pull'] = "Success"
            except Exception as e:
                print(f"[Exploration] execute_pull failed: {e}")
                exploration_results['execute_pull'] = f"Failed: {e}"

        # 5. Try execute_go (move robot)
        if 'execute_go' in available_skills and len(location_names) >= 2:
            try:
                from_location = location_names[0]
                to_location = location_names[1]
                print(f"[Exploration] Trying execute_go from {from_location} to {to_location}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=from_location,
                    to_location=to_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                exploration_results['execute_go'] = "Success"
            except Exception as e:
                print(f"[Exploration] execute_go failed: {e}")
                exploration_results['execute_go'] = f"Failed: {e}"

        # 6. Try execute_sweep (if object and location)
        if 'execute_sweep' in available_skills and test_object and test_location:
            try:
                print(f"[Exploration] Trying execute_sweep on {test_object} at {test_location}")
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    obj_name=test_object,
                    location_name=test_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                exploration_results['execute_sweep'] = "Success"
            except Exception as e:
                print(f"[Exploration] execute_sweep failed: {e}")
                exploration_results['execute_sweep'] = f"Failed: {e}"

        # 7. Try execute_gripper (no parameters)
        if 'execute_gripper' in available_skills:
            try:
                print(f"[Exploration] Trying execute_gripper")
                obs, reward, done = execute_gripper(
                    env,
                    task,
                    max_steps=100,
                    timeout=10.0
                )
                exploration_results['execute_gripper'] = "Success"
            except Exception as e:
                print(f"[Exploration] execute_gripper failed: {e}")
                exploration_results['execute_gripper'] = f"Failed: {e}"

        print("[Exploration] Exploration results:")
        for skill, result in exploration_results.items():
            print(f"  {skill}: {result}")

        # === End of Exploration Phase ===
        # Based on the results, you can analyze which predicate is missing by checking which action failed and the error message.

        # === (Optional) Main Task Plan Execution ===
        # If you have an oracle plan, you would parse and execute it step by step here using the available skills.
        # For this code, we focus on the exploration phase as per the feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()