# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects/drawers and observe the effects.

        # Get all objects and locations from positions
        object_names = [name for name in positions if positions[name].get('type', '') == 'object']
        drawer_names = [name for name in positions if positions[name].get('type', '') == 'drawer']
        location_names = [name for name in positions if positions[name].get('type', '') == 'location']

        # For this exploration, we will:
        # 1. Move to each location
        # 2. Try to pick objects on the floor
        # 3. Try to pull handles on drawers
        # 4. Try to open/close drawers
        # 5. Observe any failures or missing preconditions

        # Helper: get current robot location
        def get_robot_location():
            for name, info in positions.items():
                if info.get('is_robot', False):
                    return info.get('location', None)
            return None

        # Helper: update positions after each action
        def refresh_positions():
            return get_object_positions()

        # 1. Move to each location (using execute_go)
        robot_location = get_robot_location()
        if robot_location is None and location_names:
            robot_location = location_names[0]  # fallback

        for loc in location_names:
            if robot_location != loc:
                try:
                    print(f"[Exploration] Moving robot from {robot_location} to {loc}")
                    obs, reward, done = execute_go(env, task, robot_location, loc)
                    robot_location = loc
                except Exception as e:
                    print(f"[Exploration] execute_go failed: {e}")
            # After moving, try to pick up any object on the floor at this location
            positions = refresh_positions()
            for obj in object_names:
                obj_info = positions.get(obj, {})
                if obj_info.get('location', None) == loc and obj_info.get('on_floor', False):
                    try:
                        print(f"[Exploration] Trying to pick {obj} at {loc}")
                        obs, reward, done = execute_pick(env, task, obj, loc)
                    except Exception as e:
                        print(f"[Exploration] execute_pick failed for {obj} at {loc}: {e}")
            # Try to interact with drawers at this location
            for drawer in drawer_names:
                drawer_info = positions.get(drawer, {})
                if drawer_info.get('location', None) == loc:
                    # Try to pull the handle if possible
                    handle_name = drawer_info.get('handle', None)
                    if handle_name and handle_name in object_names:
                        try:
                            print(f"[Exploration] Trying to pick handle {handle_name} at {loc}")
                            obs, reward, done = execute_pick(env, task, handle_name, loc)
                        except Exception as e:
                            print(f"[Exploration] execute_pick failed for handle {handle_name}: {e}")
                        try:
                            print(f"[Exploration] Trying to pull drawer {drawer} with handle {handle_name} at {loc}")
                            obs, reward, done = execute_pull(env, task, drawer, handle_name, loc)
                        except Exception as e:
                            print(f"[Exploration] execute_pull failed for drawer {drawer} with handle {handle_name}: {e}")
                    # Try to open/close the drawer
                    try:
                        print(f"[Exploration] Trying to push (close) drawer {drawer} at {loc}")
                        obs, reward, done = execute_push(env, task, drawer, loc)
                    except Exception as e:
                        print(f"[Exploration] execute_push failed for drawer {drawer}: {e}")

        # After exploration, print a message
        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

        # === End of Exploration ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()