# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and try to discover which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # We will attempt to use the available skills to interact with objects and drawers,
        # and observe if any information is missing or if any action fails due to a missing predicate.

        # Get all objects, drawers, and locations from the positions dictionary
        # (Assume positions dict has keys like 'objects', 'drawers', 'locations')
        objects = positions.get('objects', [])
        drawers = positions.get('drawers', [])
        locations = positions.get('locations', [])
        handles = positions.get('handles', [])

        # For demonstration, we will try to:
        # 1. Move to each location
        # 2. Try to pick each object
        # 3. Try to open each drawer (pull)
        # 4. Try to place an object in a drawer
        # 5. Try to push (close) each drawer

        # We will log any failures or missing information to help identify the missing predicate.

        # Helper: get current robot location (if available)
        def get_robot_location():
            # Try to infer robot location from positions or observation
            # This is a placeholder; actual implementation may differ
            return positions.get('robot', None)

        # 1. Move to each location using execute_go
        robot_location = get_robot_location()
        if robot_location is None and len(locations) > 0:
            robot_location = locations[0]  # Default to first location

        for loc in locations:
            if robot_location != loc:
                try:
                    print(f"[Exploration] Moving robot from {robot_location} to {loc} using execute_go")
                    obs, reward, done = execute_go(env, task, robot_location, loc)
                    robot_location = loc
                except Exception as e:
                    print(f"[Exploration] Failed to move to {loc}: {e}")

        # 2. Try to pick each object on the floor at each location
        for obj in objects:
            for loc in locations:
                try:
                    print(f"[Exploration] Attempting to pick {obj} at {loc} using execute_pick")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Successfully picked {obj} at {loc}")
                        # Place it back (if possible) to reset state
                        # Find a drawer at this location
                        for drawer in drawers:
                            try:
                                print(f"[Exploration] Attempting to place {obj} in {drawer} at {loc} using execute_place")
                                obs, reward, done = execute_place(env, task, obj, drawer, loc)
                                if done:
                                    print(f"[Exploration] Successfully placed {obj} in {drawer} at {loc}")
                                    break
                            except Exception as e:
                                print(f"[Exploration] Failed to place {obj} in {drawer} at {loc}: {e}")
                        break
                except Exception as e:
                    print(f"[Exploration] Failed to pick {obj} at {loc}: {e}")

        # 3. Try to open each drawer (pull) using its handle at each location
        for drawer in drawers:
            for handle in handles:
                for loc in locations:
                    try:
                        print(f"[Exploration] Attempting to pick handle {handle} at {loc} for drawer {drawer}")
                        obs, reward, done = execute_pick(env, task, handle, loc)
                        if done:
                            print(f"[Exploration] Picked handle {handle} at {loc}, now trying to pull drawer {drawer}")
                            try:
                                obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                                if done:
                                    print(f"[Exploration] Successfully pulled (opened) drawer {drawer} at {loc}")
                                    # Try to push (close) the drawer
                                    try:
                                        obs, reward, done = execute_push(env, task, drawer, loc)
                                        if done:
                                            print(f"[Exploration] Successfully pushed (closed) drawer {drawer} at {loc}")
                                    except Exception as e:
                                        print(f"[Exploration] Failed to push (close) drawer {drawer} at {loc}: {e}")
                            except Exception as e:
                                print(f"[Exploration] Failed to pull (open) drawer {drawer} at {loc}: {e}")
                            # Release handle (if needed)
                            break
                    except Exception as e:
                        print(f"[Exploration] Failed to pick handle {handle} at {loc}: {e}")

        # 4. Try to sweep each object at each location
        for obj in objects:
            for loc in locations:
                try:
                    print(f"[Exploration] Attempting to sweep {obj} at {loc} using execute_sweep")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                except Exception as e:
                    print(f"[Exploration] Failed to sweep {obj} at {loc}: {e}")

        # 5. Try to use gripper (if needed)
        try:
            print(f"[Exploration] Attempting to use gripper using execute_gripper")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] Failed to use gripper: {e}")

        print("[Exploration] Exploration phase complete. Check above logs for any missing predicates or failed actions.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()