# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (drawer-closed drawer1) is missing or needs to be explored.
        # We will use available skills to explore the state of drawer1.

        # Assumptions:
        # - There is a drawer named 'drawer1' and possibly a handle object for it.
        # - The robot can move to the drawer's location and attempt to interact with it.
        # - The available skills are: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Step 1: Find the location of drawer1 and its handle (if any)
        drawer_name = 'drawer1'
        handle_name = None
        drawer_location = None

        # Try to infer handle and location from positions
        for obj_name, pos in positions.items():
            if 'handle' in obj_name and drawer_name in obj_name:
                handle_name = obj_name
            if drawer_name in obj_name:
                drawer_location = pos

        # Fallback: If not found, try to use generic names
        if drawer_location is None:
            if drawer_name in positions:
                drawer_location = positions[drawer_name]
            else:
                # Pick any location as fallback
                drawer_location = list(positions.values())[0]

        # Step 2: Move robot to the drawer's location
        # We need to know the robot's current location and the target location
        robot_current_location = None
        for obj_name, pos in positions.items():
            if 'robot' in obj_name:
                robot_current_location = pos
                break
        if robot_current_location is None:
            # Fallback: Use the first location as robot's current location
            robot_current_location = list(positions.values())[0]

        # Use execute_go to move robot to drawer location
        try:
            print(f"[Exploration] Moving robot from {robot_current_location} to {drawer_location}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_current_location,
                to_location=drawer_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")

        # Step 3: Try to interact with the drawer to determine its state
        # Try to use execute_pull (if handle is present), else try execute_push
        # This will help us determine if the drawer is closed, open, locked, etc.

        # First, try to pick the handle if available
        if handle_name is not None:
            try:
                print(f"[Exploration] Attempting to pick handle {handle_name} at {drawer_location}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=handle_name,
                    location=drawer_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Exploration] Error during execute_pick (handle): {e}")

            # Try to pull the drawer using the handle
            try:
                print(f"[Exploration] Attempting to pull drawer {drawer_name} using handle {handle_name}")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer_name=drawer_name,
                    handle_name=handle_name,
                    location=drawer_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Exploration] Error during execute_pull: {e}")
        else:
            # If no handle, try to push the drawer directly
            try:
                print(f"[Exploration] Attempting to push drawer {drawer_name} at {drawer_location}")
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer_name=drawer_name,
                    location=drawer_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Exploration] Error during execute_push: {e}")

        # Step 4: After interaction, check observation/state for drawer-closed predicate
        # (This would typically be done by inspecting the environment state or observation)
        # For demonstration, print out the observation
        try:
            obs = task.get_observation()
            print(f"[Exploration] Observation after interaction: {obs}")
            # If the observation includes predicates, check for (drawer-closed drawer1)
            if hasattr(obs, 'predicates'):
                if ('drawer-closed', drawer_name) in obs.predicates:
                    print(f"[Exploration] Predicate (drawer-closed {drawer_name}) is TRUE.")
                else:
                    print(f"[Exploration] Predicate (drawer-closed {drawer_name}) is FALSE or missing.")
            else:
                print("[Exploration] No predicate information available in observation.")
        except Exception as e:
            print(f"[Exploration] Error retrieving observation: {e}")

        # === End of Exploration Phase ===

        # (Optional) Continue with further task logic or planning as needed

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()