# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available skills to explore the environment and try to discover missing predicates.
        # Since we cannot define new skills, we use the available ones to interact and observe.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to explore the environment using available skills
        # For demonstration, we will attempt to use each skill in a safe, exploratory way
        # and print out any exceptions or observations that may indicate missing predicates.

        # Get all objects and locations from positions
        object_names = []
        location_names = []
        drawer_names = []
        handle_names = []
        for name, pos in positions.items():
            if 'drawer' in name:
                drawer_names.append(name)
            elif 'handle' in name:
                handle_names.append(name)
            elif 'loc' in name or 'location' in name or 'room' in name:
                location_names.append(name)
            else:
                object_names.append(name)

        # Fallback: If no explicit locations, use unique positions as locations
        if not location_names:
            unique_locs = set()
            for pos in positions.values():
                unique_locs.add(tuple(np.round(pos, 2)))
            location_names = [f"loc_{i}" for i in range(len(unique_locs))]

        # Try to move the robot to each location (if possible)
        robot_location = None
        try:
            # Try to infer robot's current location from observation or positions
            if hasattr(task, 'get_robot_location'):
                robot_location = task.get_robot_location()
            else:
                # Fallback: pick the first location
                robot_location = location_names[0] if location_names else None
        except Exception as e:
            robot_location = None

        # Exploration: Try to use each skill and catch errors to infer missing predicates
        for skill in available_skills:
            try:
                if skill == 'execute_go':
                    # Try moving between two locations
                    if len(location_names) >= 2:
                        from_loc = location_names[0]
                        to_loc = location_names[1]
                        print(f"[Exploration] Trying {skill} from {from_loc} to {to_loc}")
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                elif skill == 'execute_pick':
                    # Try picking up an object from the floor
                    if object_names and location_names:
                        obj = object_names[0]
                        loc = location_names[0]
                        print(f"[Exploration] Trying {skill} on {obj} at {loc}")
                        obs, reward, done = execute_pick(env, task, obj, loc)
                elif skill == 'execute_place':
                    # Try placing an object into a drawer
                    if object_names and drawer_names and location_names:
                        obj = object_names[0]
                        drawer = drawer_names[0]
                        loc = location_names[0]
                        print(f"[Exploration] Trying {skill} for {obj} into {drawer} at {loc}")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                elif skill == 'execute_push':
                    # Try pushing (closing) a drawer
                    if drawer_names and location_names:
                        drawer = drawer_names[0]
                        loc = location_names[0]
                        print(f"[Exploration] Trying {skill} on {drawer} at {loc}")
                        obs, reward, done = execute_push(env, task, drawer, loc)
                elif skill == 'execute_pull':
                    # Try pulling (opening) a drawer using a handle
                    if drawer_names and handle_names and location_names:
                        drawer = drawer_names[0]
                        handle = handle_names[0]
                        loc = location_names[0]
                        print(f"[Exploration] Trying {skill} on {drawer} with {handle} at {loc}")
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                elif skill == 'execute_sweep':
                    # Try sweeping an object
                    if object_names and location_names:
                        obj = object_names[0]
                        loc = location_names[0]
                        print(f"[Exploration] Trying {skill} on {obj} at {loc}")
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                elif skill == 'execute_rotate':
                    # Try rotating (if implemented)
                    print(f"[Exploration] Trying {skill} (no parameters)")
                    obs, reward, done = execute_rotate(env, task)
                elif skill == 'execute_gripper':
                    # Try gripper action (if implemented)
                    print(f"[Exploration] Trying {skill} (no parameters)")
                    obs, reward, done = execute_gripper(env, task)
            except Exception as e:
                print(f"[Exploration] {skill} failed: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

        # === End of Exploration Phase ===

        # === (Optional) Main Task Plan Execution ===
        # At this point, after exploration, you would normally execute the oracle plan.
        # Since the plan cannot be executed due to missing predicates, we stop here.
        # If the missing predicate is discovered and fixed, you can insert the plan execution code below.

        # Example (commented out):
        # obs, reward, done = execute_pick(env, task, 'object_1', 'location_1')
        # if done:
        #     print("[Task] Task ended after picking object_1!")
        #     return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()