# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (drawer-unlocked) ===
        # Feedback indicates (drawer-unlocked drawer1) is missing.
        # We need to explore and check the lock state of the drawer before attempting to open it.
        # The exploration domain suggests that pulling the handle can reveal lock status.

        # For this example, we assume the following object names and structure:
        # - drawer1: the drawer to manipulate
        # - handle1: the handle attached to drawer1
        # - object1: the object to pick and place
        # - locations: 'loc_robot', 'loc_drawer', etc.

        # You may need to adjust these names based on your actual environment.
        # For demonstration, we use generic names.

        # --- Example object and location names (replace with actual names as needed) ---
        drawer_name = 'drawer1'
        handle_name = 'handle1'
        object_name = 'object1'
        robot_location = 'loc_robot'
        drawer_location = 'loc_drawer'
        object_location = 'loc_object'

        # Try to get actual positions if available
        try:
            drawer_pos = positions.get(drawer_name, None)
            handle_pos = positions.get(handle_name, None)
            object_pos = positions.get(object_name, None)
        except Exception as e:
            print("[Warning] Could not retrieve some object positions:", e)
            drawer_pos = handle_pos = object_pos = None

        # === Step 1: Move robot to drawer location (if not already there) ===
        try:
            # Use execute_go to move robot from current location to drawer location
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=drawer_location
            )
            print("[Exploration] Robot moved to drawer location.")
        except Exception as e:
            print("[Error] Failed to move robot to drawer location:", e)

        # === Step 2: Exploration - Try to pull the handle to check lock status ===
        # This will help us discover if the drawer is locked or unlocked.
        # If the drawer is locked, the pull will fail or indicate lock-known.
        # If the drawer is unlocked, the pull will succeed.

        # First, pick the handle (if required by your domain)
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                obj=handle_name,
                location=drawer_location
            )
            print("[Exploration] Picked up the handle to check lock status.")
        except Exception as e:
            print("[Error] Failed to pick handle for exploration:", e)

        # Now, try to pull the handle (exploration for lock-known)
        try:
            obs, reward, done = execute_pull(
                env,
                task,
                drawer=drawer_name,
                handle=handle_name,
                location=drawer_location
            )
            print("[Exploration] Pulled the handle to check if drawer is unlocked.")
        except Exception as e:
            print("[Exploration] Pull failed, possibly due to locked drawer or missing predicate:", e)
            print("[Exploration] The missing predicate is likely (drawer-unlocked drawer1).")

        # === Step 3: If drawer is unlocked, proceed with the main plan ===
        # For demonstration, we continue with a typical pick-and-place sequence.

        # 3.1 Move to object location (if needed)
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=drawer_location,
                to_location=object_location
            )
            print("[Task] Robot moved to object location.")
        except Exception as e:
            print("[Error] Failed to move to object location:", e)

        # 3.2 Pick the object
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                obj=object_name,
                location=object_location
            )
            print("[Task] Picked up the object.")
        except Exception as e:
            print("[Error] Failed to pick the object:", e)

        # 3.3 Move back to drawer location
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=object_location,
                to_location=drawer_location
            )
            print("[Task] Returned to drawer location.")
        except Exception as e:
            print("[Error] Failed to return to drawer location:", e)

        # 3.4 Place the object in the drawer (assume drawer is open)
        try:
            obs, reward, done = execute_place(
                env,
                task,
                obj=object_name,
                drawer=drawer_name,
                location=drawer_location
            )
            print("[Task] Placed the object in the drawer.")
        except Exception as e:
            print("[Error] Failed to place the object in the drawer:", e)

        # 3.5 Push to close the drawer
        try:
            obs, reward, done = execute_push(
                env,
                task,
                drawer=drawer_name,
                location=drawer_location
            )
            print("[Task] Closed the drawer.")
        except Exception as e:
            print("[Error] Failed to close the drawer:", e)

        # === End of Task ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()