# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback suggests we need to check if the object is in the object list before inserting into the drawer.
        # We also need to ensure the drawer is open and accessible.
        # We'll perform an exploration step to check for object presence and drawer state.

        # For demonstration, let's assume the following object and drawer names (replace with actual names as needed)
        object_list = [name for name in positions.keys() if "object" in name or "dice" in name or "ball" in name]
        drawer_list = [name for name in positions.keys() if "drawer" in name]
        handle_list = [name for name in positions.keys() if "handle" in name]

        # For this example, let's pick the first object and first drawer
        if not object_list or not drawer_list:
            print("[Error] No objects or drawers found in the environment.")
            return

        target_object = object_list[0]
        target_drawer = drawer_list[0]
        # Try to find a handle for the drawer
        target_handle = None
        for h in handle_list:
            if target_drawer in h:
                target_handle = h
                break
        if not target_handle and handle_list:
            target_handle = handle_list[0]  # fallback

        # Get locations (assume positions dict gives location names or coordinates)
        # For this code, we assume location names are available
        # If positions[target_object] is a tuple, you may need to map it to a location name
        # For now, we use the keys as location names
        robot_location = None
        for k in positions:
            if "robot" in k:
                robot_location = k
                break
        if not robot_location:
            robot_location = "robot_base"  # fallback

        object_location = None
        if isinstance(positions[target_object], str):
            object_location = positions[target_object]
        else:
            object_location = "table"  # fallback

        drawer_location = None
        if isinstance(positions[target_drawer], str):
            drawer_location = positions[target_drawer]
        else:
            drawer_location = "drawer_area"  # fallback

        # === Exploration: Check if object is in the object list and drawer is open ===
        print(f"[Exploration] Checking if {target_object} is in object list and {target_drawer} is open.")

        if target_object not in object_list:
            print(f"[Exploration] {target_object} is not in the object list. Aborting insertion.")
            return

        # Here, we would check the state of the drawer (open/closed)
        # For this example, let's assume we can get this info from the environment or positions
        # If not, we proceed to open the drawer

        # === Plan Execution ===

        # 1. Move robot to object location if not already there
        try:
            print(f"[Task] Moving robot to object location: {object_location}")
            obs, reward, done = execute_go(env, task, from_location=robot_location, to_location=object_location)
            robot_location = object_location
        except Exception as e:
            print(f"[Error] Failed to move to object location: {e}")
            return

        # 2. Pick up the object
        try:
            print(f"[Task] Picking up object: {target_object} at {object_location}")
            obs, reward, done = execute_pick(env, task, obj=target_object, location=object_location)
        except Exception as e:
            print(f"[Error] Failed to pick up object: {e}")
            return

        # 3. Move to drawer location
        try:
            print(f"[Task] Moving robot to drawer location: {drawer_location}")
            obs, reward, done = execute_go(env, task, from_location=object_location, to_location=drawer_location)
            robot_location = drawer_location
        except Exception as e:
            print(f"[Error] Failed to move to drawer location: {e}")
            return

        # 4. Open the drawer if not already open
        # For this example, we always try to open
        try:
            print(f"[Task] Pulling handle {target_handle} to open drawer {target_drawer}")
            obs, reward, done = execute_pull(env, task, drawer=target_drawer, handle=target_handle, location=drawer_location)
        except Exception as e:
            print(f"[Error] Failed to open drawer: {e}")
            return

        # 5. Place the object into the drawer
        try:
            print(f"[Task] Placing object {target_object} into drawer {target_drawer} at {drawer_location}")
            obs, reward, done = execute_place(env, task, obj=target_object, drawer=target_drawer, location=drawer_location)
        except Exception as e:
            print(f"[Error] Failed to place object into drawer: {e}")
            return

        # 6. Push (close) the drawer
        try:
            print(f"[Task] Pushing drawer {target_drawer} closed at {drawer_location}")
            obs, reward, done = execute_push(env, task, drawer=target_drawer, location=drawer_location)
        except Exception as e:
            print(f"[Error] Failed to close drawer: {e}")
            return

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
