# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available skills to explore the environment and try to discover missing predicates.
        # For this, we attempt to execute all available skills in a safe exploratory manner.

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to execute each skill with plausible arguments to see if any precondition fails
        # or if the environment provides feedback about missing predicates.
        # This is a diagnostic phase to help the user identify what is missing.

        print("=== Exploration Phase: Checking for missing predicates via skill execution ===")
        for skill_name in available_skills:
            try:
                print(f"[Exploration] Attempting skill: {skill_name}")
                if skill_name == 'execute_go':
                    # Try moving between two locations if available
                    locs = [k for k, v in positions.items() if 'location' in k or 'room' in k or 'pos' in k]
                    if len(locs) >= 2:
                        from_loc = locs[0]
                        to_loc = locs[1]
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        print(f"Executed {skill_name} from {from_loc} to {to_loc}")
                elif skill_name == 'execute_pick':
                    # Try picking up an object on the floor
                    objs = [k for k, v in positions.items() if 'object' in k or 'obj' in k or 'ball' in k or 'handle' in k]
                    locs = [k for k, v in positions.items() if 'location' in k or 'room' in k or 'pos' in k]
                    if objs and locs:
                        obj = objs[0]
                        loc = locs[0]
                        obs, reward, done = execute_pick(env, task, obj, loc)
                        print(f"Executed {skill_name} on {obj} at {loc}")
                elif skill_name == 'execute_place':
                    # Try placing an object into a drawer at a location
                    objs = [k for k, v in positions.items() if 'object' in k or 'obj' in k or 'ball' in k]
                    drawers = [k for k, v in positions.items() if 'drawer' in k]
                    locs = [k for k, v in positions.items() if 'location' in k or 'room' in k or 'pos' in k]
                    if objs and drawers and locs:
                        obj = objs[0]
                        drawer = drawers[0]
                        loc = locs[0]
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        print(f"Executed {skill_name} for {obj} into {drawer} at {loc}")
                elif skill_name == 'execute_push':
                    # Try pushing a drawer closed
                    drawers = [k for k, v in positions.items() if 'drawer' in k]
                    locs = [k for k, v in positions.items() if 'location' in k or 'room' in k or 'pos' in k]
                    if drawers and locs:
                        drawer = drawers[0]
                        loc = locs[0]
                        obs, reward, done = execute_push(env, task, drawer, loc)
                        print(f"Executed {skill_name} on {drawer} at {loc}")
                elif skill_name == 'execute_pull':
                    # Try pulling a handle to open a drawer
                    drawers = [k for k, v in positions.items() if 'drawer' in k]
                    handles = [k for k, v in positions.items() if 'handle' in k]
                    locs = [k for k, v in positions.items() if 'location' in k or 'room' in k or 'pos' in k]
                    if drawers and handles and locs:
                        drawer = drawers[0]
                        handle = handles[0]
                        loc = locs[0]
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        print(f"Executed {skill_name} on {drawer} with {handle} at {loc}")
                elif skill_name == 'execute_sweep':
                    # Try sweeping an object at a location
                    objs = [k for k, v in positions.items() if 'object' in k or 'obj' in k or 'ball' in k]
                    locs = [k for k, v in positions.items() if 'location' in k or 'room' in k or 'pos' in k]
                    if objs and locs:
                        obj = objs[0]
                        loc = locs[0]
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                        print(f"Executed {skill_name} on {obj} at {loc}")
                elif skill_name == 'execute_rotate':
                    # If rotate skill exists, try to rotate something (not enough info, skip or log)
                    print(f"[Exploration] {skill_name} not implemented in this context.")
                elif skill_name == 'execute_gripper':
                    # Try gripper action (no parameters)
                    obs, reward, done = execute_gripper(env, task)
                    print(f"Executed {skill_name}")
            except Exception as e:
                print(f"[Exploration] Skill {skill_name} failed: {e}")

        print("=== Exploration Phase Complete ===")
        print("If any skill failed due to a missing predicate, check the error messages above for clues.")

        # === Main Task Plan Execution ===
        # At this point, after exploration, you would normally execute the oracle plan.
        # Since the plan is not provided, and the main issue is missing predicates,
        # the code above helps you identify which predicate is missing by attempting all skills.

        # If you have a plan, you would execute it step by step here using the same skill functions.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()