# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We use the available skills to explore the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # We will attempt to use the available skills to interact with objects and drawers,
        # and observe if any expected state change or feedback is missing, indicating a missing predicate.

        # Get all objects and locations from the environment
        object_names = [name for name in positions.keys() if 'object' in name or 'obj' in name]
        drawer_names = [name for name in positions.keys() if 'drawer' in name]
        location_names = [name for name in positions.keys() if 'location' in name or 'loc' in name or 'room' in name]

        # Fallback: If no explicit location names, use all unique positions as locations
        if not location_names:
            location_names = list(set([positions[name] for name in positions]))

        # Try to execute each available skill and observe the result
        # This is a simple exploration loop to probe the environment for missing predicates

        # 1. Try to pick up each object from the floor
        for obj in object_names:
            try:
                obj_pos = positions[obj]
                print(f"[Exploration] Attempting to pick up {obj} at {obj_pos}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj,
                    obj_pos
                )
                if done:
                    print(f"[Exploration] Successfully picked up {obj}.")
                else:
                    print(f"[Exploration] Could not pick up {obj}. Possible missing predicate or precondition.")
            except Exception as e:
                print(f"[Exploration] Exception during execute_pick for {obj}: {e}")

        # 2. Try to open each drawer using its handle (if handle is known)
        for drawer in drawer_names:
            # Find handle for this drawer
            handle_name = None
            for obj in object_names:
                if f"handle_of_{drawer}" in obj or f"{drawer}_handle" in obj or "handle" in obj:
                    handle_name = obj
                    break
            if handle_name is None:
                continue  # No handle found for this drawer
            try:
                drawer_pos = positions[drawer]
                print(f"[Exploration] Attempting to pick up handle {handle_name} for drawer {drawer}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    handle_name,
                    drawer_pos
                )
                if not done:
                    print(f"[Exploration] Could not pick up handle {handle_name}.")
                    continue
                print(f"[Exploration] Attempting to pull open drawer {drawer} with handle {handle_name}")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer,
                    handle_name,
                    drawer_pos
                )
                if done:
                    print(f"[Exploration] Successfully pulled open drawer {drawer}.")
                else:
                    print(f"[Exploration] Could not pull open drawer {drawer}. Possible missing predicate (e.g., lock-known, drawer-unlocked, drawer-closed).")
            except Exception as e:
                print(f"[Exploration] Exception during execute_pull for {drawer}: {e}")

        # 3. Try to place an object into a drawer (if possible)
        for obj in object_names:
            for drawer in drawer_names:
                try:
                    drawer_pos = positions[drawer]
                    print(f"[Exploration] Attempting to place {obj} into {drawer} at {drawer_pos}")
                    obs, reward, done = execute_place(
                        env,
                        task,
                        obj,
                        drawer,
                        drawer_pos
                    )
                    if done:
                        print(f"[Exploration] Successfully placed {obj} into {drawer}.")
                    else:
                        print(f"[Exploration] Could not place {obj} into {drawer}. Possible missing predicate (e.g., drawer-open, drawer-full, holding).")
                except Exception as e:
                    print(f"[Exploration] Exception during execute_place for {obj} into {drawer}: {e}")

        # 4. Try to move the robot between locations (if possible)
        if len(location_names) >= 2:
            from_loc = location_names[0]
            to_loc = location_names[1]
            try:
                print(f"[Exploration] Attempting to move robot from {from_loc} to {to_loc}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_loc,
                    to_loc
                )
                if done:
                    print(f"[Exploration] Robot moved from {from_loc} to {to_loc}.")
                else:
                    print(f"[Exploration] Could not move robot from {from_loc} to {to_loc}. Possible missing predicate (e.g., robot-free, robot-at).")
            except Exception as e:
                print(f"[Exploration] Exception during execute_go: {e}")

        # 5. Try to push drawers closed (if possible)
        for drawer in drawer_names:
            try:
                drawer_pos = positions[drawer]
                print(f"[Exploration] Attempting to push drawer {drawer} closed at {drawer_pos}")
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer,
                    drawer_pos
                )
                if done:
                    print(f"[Exploration] Successfully pushed drawer {drawer} closed.")
                else:
                    print(f"[Exploration] Could not push drawer {drawer} closed. Possible missing predicate (e.g., drawer-open, robot-free).")
            except Exception as e:
                print(f"[Exploration] Exception during execute_push for {drawer}: {e}")

        # 6. Try to sweep objects (if available)
        for obj in object_names:
            try:
                obj_pos = positions[obj]
                print(f"[Exploration] Attempting to sweep {obj} at {obj_pos}")
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    obj,
                    obj_pos
                )
                if done:
                    print(f"[Exploration] Successfully swept {obj}.")
                else:
                    print(f"[Exploration] Could not sweep {obj}.")
            except Exception as e:
                print(f"[Exploration] Exception during execute_sweep for {obj}: {e}")

        # 7. Try to use gripper (if available)
        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(
                env,
                task
            )
            if done:
                print(f"[Exploration] Successfully used gripper.")
            else:
                print(f"[Exploration] Could not use gripper.")
        except Exception as e:
            print(f"[Exploration] Exception during execute_gripper: {e}")

        print("[Exploration] Exploration phase complete. Review the above logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()