# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate (robot-free) ===
        # The feedback indicates that (robot-free) is missing or not being set as expected.
        # We will perform a sequence of actions to explore the environment and check the robot's state.

        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We will attempt to use the available skills to reach a state where (robot-free) is true.
        # Since we do not have the exact object and location names, we will use the positions dictionary to infer them.

        # Step 1: List all objects and locations
        object_names = []
        location_names = []
        drawer_names = []
        handle_names = []
        for name, pos in positions.items():
            if "drawer" in name:
                drawer_names.append(name)
            elif "handle" in name:
                handle_names.append(name)
            elif "loc" in name or "location" in name or "room" in name:
                location_names.append(name)
            else:
                object_names.append(name)

        # Fallback: If no explicit locations, use all unique positions as locations
        if not location_names:
            location_names = list(set([tuple(pos) for pos in positions.values()]))

        # Step 2: Try to execute a pick and then a place to see if robot-free is set
        # We will pick the first object on the floor and place it in the first drawer (if available)
        # If not, we will just execute pick and then push/pull to explore the state

        # Find an object and a location
        picked = False
        placed = False
        pushed = False
        pulled = False

        # Try to pick an object
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to pick {obj} at {loc}")
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        obj,
                        loc
                    )
                    picked = True
                    break
                except Exception as e:
                    print(f"[Exploration] Failed to pick {obj} at {loc}: {e}")
            if picked:
                break

        # Try to place the object in a drawer if possible
        if picked and drawer_names:
            for drawer in drawer_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Trying to place {obj} in {drawer} at {loc}")
                        obs, reward, done = execute_place(
                            env,
                            task,
                            obj,
                            drawer,
                            loc
                        )
                        placed = True
                        break
                    except Exception as e:
                        print(f"[Exploration] Failed to place {obj} in {drawer} at {loc}: {e}")
                if placed:
                    break

        # Try to push a drawer (close it)
        if drawer_names:
            for drawer in drawer_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Trying to push (close) {drawer} at {loc}")
                        obs, reward, done = execute_push(
                            env,
                            task,
                            drawer,
                            loc
                        )
                        pushed = True
                        break
                    except Exception as e:
                        print(f"[Exploration] Failed to push {drawer} at {loc}: {e}")
                if pushed:
                    break

        # Try to pull a drawer (open it) using a handle if available
        if drawer_names and handle_names:
            for drawer in drawer_names:
                for handle in handle_names:
                    for loc in location_names:
                        try:
                            print(f"[Exploration] Trying to pull (open) {drawer} with {handle} at {loc}")
                            obs, reward, done = execute_pull(
                                env,
                                task,
                                drawer,
                                handle,
                                loc
                            )
                            pulled = True
                            break
                        except Exception as e:
                            print(f"[Exploration] Failed to pull {drawer} with {handle} at {loc}: {e}")
                    if pulled:
                        break
                if pulled:
                    break

        # Try to move the robot between locations
        if len(location_names) >= 2:
            from_loc = location_names[0]
            to_loc = location_names[1]
            try:
                print(f"[Exploration] Trying to move robot from {from_loc} to {to_loc}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_loc,
                    to_loc
                )
            except Exception as e:
                print(f"[Exploration] Failed to move robot from {from_loc} to {to_loc}: {e}")

        # Try to sweep an object
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to sweep {obj} at {loc}")
                    obs, reward, done = execute_sweep(
                        env,
                        task,
                        obj,
                        loc
                    )
                    break
                except Exception as e:
                    print(f"[Exploration] Failed to sweep {obj} at {loc}: {e}")
            break

        # Try to use the gripper
        try:
            print(f"[Exploration] Trying to execute gripper")
            obs, reward, done = execute_gripper(
                env,
                task
            )
        except Exception as e:
            print(f"[Exploration] Failed to execute gripper: {e}")

        # After exploration, print a message about the missing predicate
        print("[Exploration] Exploration phase complete. If (robot-free) is not set, it may be missing from the domain or not being triggered by the available skills.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()