# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (drawer-unlocked drawer1) is missing in the initial state.
        # We need to explore to determine the lock state of drawer1.
        # The exploration domain suggests that "lock-known" is the predicate to be discovered.
        # We will use the available skills to perform exploration.

        # Assumptions for object and location names (update as needed for your environment)
        drawer_name = 'drawer1'
        handle_name = 'handle1'
        robot_location = None
        drawer_location = None
        handle_location = None

        # Try to infer locations from positions dictionary
        for obj, pos in positions.items():
            if 'robot' in obj:
                robot_location = obj
            if drawer_name in obj:
                drawer_location = obj
            if handle_name in obj:
                handle_location = obj

        # Fallbacks if not found
        if robot_location is None:
            robot_location = 'robot_base'
        if drawer_location is None:
            drawer_location = drawer_name
        if handle_location is None:
            handle_location = handle_name

        print(f"[Exploration] Robot at: {robot_location}, Drawer: {drawer_location}, Handle: {handle_location}")

        # 1. Move robot to the drawer location if not already there
        try:
            if robot_location != drawer_location:
                print(f"[Exploration] Moving robot from {robot_location} to {drawer_location}")
                obs, reward, done, info = execute_go(env, task, robot_location, drawer_location)
                robot_location = drawer_location
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")

        # 2. Pick the handle to prepare for pulling (if required by domain)
        try:
            print(f"[Exploration] Picking handle: {handle_name} at {drawer_location}")
            obs, reward, done, info = execute_pick(env, task, handle_name, drawer_location)
        except Exception as e:
            print(f"[Exploration] Error during execute_pick: {e}")

        # 3. Attempt to pull the drawer to discover its lock state
        try:
            print(f"[Exploration] Attempting to pull drawer: {drawer_name} using handle: {handle_name}")
            obs, reward, done, info = execute_pull(env, task, drawer_name, handle_name, drawer_location)
            # If the drawer is unlocked, this should succeed and the predicate (drawer-unlocked drawer1) is confirmed
            print("[Exploration] Drawer pull attempted. If successful, drawer is unlocked.")
        except Exception as e:
            print(f"[Exploration] Error during execute_pull: {e}")
            print("[Exploration] Drawer may be locked or another issue occurred.")

        # At this point, the missing predicate (drawer-unlocked drawer1) should be discovered or confirmed

        # === Main Task Plan Execution ===
        # Insert your oracle plan here, using only the available skills.
        # For demonstration, we show a generic sequence (update as needed for your task):

        # Example: Place an object into the drawer after confirming it is unlocked and open
        object_to_place = 'object1'
        try:
            # 4. Pick up the object from the floor
            print(f"[Task] Picking up {object_to_place} from floor at {drawer_location}")
            obs, reward, done, info = execute_pick(env, task, object_to_place, drawer_location)
        except Exception as e:
            print(f"[Task] Error during execute_pick for {object_to_place}: {e}")

        try:
            # 5. Place the object into the drawer
            print(f"[Task] Placing {object_to_place} into {drawer_name} at {drawer_location}")
            obs, reward, done, info = execute_place(env, task, object_to_place, drawer_name, drawer_location)
        except Exception as e:
            print(f"[Task] Error during execute_place for {object_to_place}: {e}")

        try:
            # 6. Push (close) the drawer
            print(f"[Task] Pushing (closing) {drawer_name} at {drawer_location}")
            obs, reward, done, info = execute_push(env, task, drawer_name, drawer_location)
        except Exception as e:
            print(f"[Task] Error during execute_push for {drawer_name}: {e}")

        # === End of Task ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()