# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (drawer-closed drawer1) is missing or needs to be checked.
        # We will attempt to use available skills to explore the state of drawer1.

        # Assumptions for object names and locations (update as per your environment)
        drawer_name = 'drawer1'
        handle_name = 'handle1'
        robot_location = 'loc_robot'
        drawer_location = 'loc_drawer'
        object_to_pick = 'obj1'  # Example object on floor
        place_location = drawer_location  # Place near drawer

        # Try to get actual positions if available
        try:
            drawer_pos = positions.get(drawer_name, None)
            handle_pos = positions.get(handle_name, None)
            robot_pos = positions.get('robot', None)
            object_pos = positions.get(object_to_pick, None)
        except Exception as e:
            print("[Exploration] Could not retrieve some positions:", e)
            drawer_pos = handle_pos = robot_pos = object_pos = None

        # Step 1: Move robot to drawer location to explore its state
        try:
            print("[Exploration] Moving robot to drawer location to check state...")
            # Use execute_go to move robot (if not already there)
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=drawer_location
            )
        except Exception as e:
            print("[Exploration] Error during execute_go:", e)

        # Step 2: Try to pull the drawer to see if it is closed/locked/unlocked
        # This will help us determine if (drawer-closed drawer1) is true/missing
        try:
            print("[Exploration] Attempting to pick handle and pull drawer...")
            # Pick the handle first (assume handle is an object on the drawer)
            obs, reward, done = execute_pick(
                env,
                task,
                obj=handle_name,
                location=drawer_location
            )
            # Try to pull the drawer using the handle
            obs, reward, done = execute_pull(
                env,
                task,
                drawer=drawer_name,
                h=handle_name,
                p=drawer_location
            )
            print("[Exploration] Pull action executed. If drawer was closed and unlocked, it should now be open.")
        except Exception as e:
            print("[Exploration] Error during execute_pick or execute_pull:", e)
            print("[Exploration] This may indicate the drawer is not closed, not unlocked, or handle is missing.")

        # Step 3: If pull fails, try to push (close) the drawer to ensure it is closed
        try:
            print("[Exploration] Attempting to push (close) the drawer...")
            obs, reward, done = execute_push(
                env,
                task,
                d=drawer_name,
                p=drawer_location
            )
            print("[Exploration] Push action executed. Drawer should now be closed.")
        except Exception as e:
            print("[Exploration] Error during execute_push:", e)
            print("[Exploration] Drawer may already be closed or not open.")

        # Step 4: Now, try to pull again to confirm the closed state
        try:
            print("[Exploration] Re-attempting to pick handle and pull drawer after push...")
            obs, reward, done = execute_pick(
                env,
                task,
                obj=handle_name,
                location=drawer_location
            )
            obs, reward, done = execute_pull(
                env,
                task,
                drawer=drawer_name,
                h=handle_name,
                p=drawer_location
            )
            print("[Exploration] Pull action executed after push. If successful, drawer was closed and is now open.")
        except Exception as e:
            print("[Exploration] Error during second execute_pick or execute_pull:", e)
            print("[Exploration] This may confirm the drawer was closed before.")

        # === Main Task Plan (Example) ===
        # Now that we have explored the drawer state, proceed with the main plan.
        # For demonstration, pick an object and place it in the drawer.

        try:
            print("[Task] Moving robot to object location...")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=drawer_location,
                to_location=object_pos if object_pos else place_location
            )
        except Exception as e:
            print("[Task] Error during execute_go to object:", e)

        try:
            print("[Task] Picking up object...")
            obs, reward, done = execute_pick(
                env,
                task,
                obj=object_to_pick,
                location=object_pos if object_pos else place_location
            )
        except Exception as e:
            print("[Task] Error during execute_pick:", e)

        try:
            print("[Task] Moving robot to drawer location for placing object...")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=object_pos if object_pos else place_location,
                to_location=drawer_location
            )
        except Exception as e:
            print("[Task] Error during execute_go to drawer:", e)

        try:
            print("[Task] Placing object in drawer...")
            obs, reward, done = execute_place(
                env,
                task,
                o=object_to_pick,
                d=drawer_name,
                p=drawer_location
            )
        except Exception as e:
            print("[Task] Error during execute_place:", e)

        print("[Task] Task execution complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()