# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (robot-free) is missing or problematic.
        # We will attempt to execute skills that require (robot-free) and observe the result.
        # If an action fails due to missing (robot-free), we log this and continue exploration.

        # For demonstration, we will try to execute each available skill in a safe, exploratory way.
        # The goal is to determine if (robot-free) is required and missing in the current state.

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For exploration, we need to select plausible arguments for each skill.
        # We'll use the first available object, drawer, handle, and location from positions.
        # This is a generic approach; in a real scenario, you would parse the observation/init.

        # Extract example objects for exploration
        object_names = [name for name in positions if 'object' in name]
        drawer_names = [name for name in positions if 'drawer' in name]
        handle_names = [name for name in positions if 'handle' in name]
        location_names = [name for name in positions if 'location' in name or 'room' in name or 'pos' in name]

        # Fallbacks in case names are not found
        example_object = object_names[0] if object_names else None
        example_drawer = drawer_names[0] if drawer_names else None
        example_handle = handle_names[0] if handle_names else None
        example_location = location_names[0] if location_names else None

        # For execute_go, we need two locations
        if len(location_names) >= 2:
            from_location = location_names[0]
            to_location = location_names[1]
        else:
            from_location = example_location
            to_location = example_location

        # Exploration log
        exploration_results = {}

        print("=== Exploration Phase: Testing Skills for Missing Predicate (robot-free) ===")
        for skill in available_skills:
            try:
                print(f"[Exploration] Attempting skill: {skill}")
                if skill == 'execute_pick':
                    if example_object and example_location:
                        obs, reward, done = execute_pick(
                            env, task, obj=example_object, location=example_location
                        )
                        exploration_results[skill] = "Success"
                    else:
                        exploration_results[skill] = "Skipped (missing object/location)"
                elif skill == 'execute_place':
                    if example_object and example_drawer and example_location:
                        obs, reward, done = execute_place(
                            env, task, obj=example_object, drawer=example_drawer, location=example_location
                        )
                        exploration_results[skill] = "Success"
                    else:
                        exploration_results[skill] = "Skipped (missing object/drawer/location)"
                elif skill == 'execute_push':
                    if example_drawer and example_location:
                        obs, reward, done = execute_push(
                            env, task, drawer=example_drawer, location=example_location
                        )
                        exploration_results[skill] = "Success"
                    else:
                        exploration_results[skill] = "Skipped (missing drawer/location)"
                elif skill == 'execute_pull':
                    if example_drawer and example_handle and example_location:
                        obs, reward, done = execute_pull(
                            env, task, drawer=example_drawer, handle=example_handle, location=example_location
                        )
                        exploration_results[skill] = "Success"
                    else:
                        exploration_results[skill] = "Skipped (missing drawer/handle/location)"
                elif skill == 'execute_sweep':
                    if example_object and example_location:
                        obs, reward, done = execute_sweep(
                            env, task, obj=example_object, location=example_location
                        )
                        exploration_results[skill] = "Success"
                    else:
                        exploration_results[skill] = "Skipped (missing object/location)"
                elif skill == 'execute_rotate':
                    # No parameters specified in domain, so just call with env, task
                    obs, reward, done = execute_rotate(env, task)
                    exploration_results[skill] = "Success"
                elif skill == 'execute_go':
                    if from_location and to_location:
                        obs, reward, done = execute_go(
                            env, task, from_location=from_location, to_location=to_location
                        )
                        exploration_results[skill] = "Success"
                    else:
                        exploration_results[skill] = "Skipped (missing locations)"
                elif skill == 'execute_gripper':
                    obs, reward, done = execute_gripper(env, task)
                    exploration_results[skill] = "Success"
                else:
                    exploration_results[skill] = "Unknown skill"
            except Exception as e:
                # If the error message mentions 'robot-free', log it as a missing predicate
                msg = str(e)
                if 'robot-free' in msg or 'robot free' in msg:
                    print(f"[Exploration] Skill {skill} failed due to missing (robot-free) predicate.")
                    exploration_results[skill] = "Failed: missing (robot-free)"
                else:
                    print(f"[Exploration] Skill {skill} failed: {msg}")
                    exploration_results[skill] = f"Failed: {msg}"

        print("=== Exploration Results ===")
        for skill, result in exploration_results.items():
            print(f"  {skill}: {result}")

        # === End of Exploration Phase ===

        # TODO: After identifying the missing predicate, you would update the domain or initial state accordingly.
        # For this task, the exploration phase is the main requirement.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()