# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We use the available skills to explore the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We'll attempt to use the available skills to interact with objects/drawers and observe the environment's response.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to explore all objects and drawers to find missing predicates
        # For demonstration, we attempt to pull on all handles to check for lock-known predicate
        # and pick/place objects to check for other missing predicates.

        # Get all objects and drawers from positions (assuming keys are object/drawer names)
        object_names = [name for name in positions.keys() if 'object' in name]
        drawer_names = [name for name in positions.keys() if 'drawer' in name]
        handle_names = [name for name in positions.keys() if 'handle' in name]
        location_names = [name for name in positions.keys() if 'location' in name or 'pos' in name]

        # For robustness, collect all unique locations
        all_locations = set()
        for pos in positions.values():
            if isinstance(pos, tuple) and len(pos) == 3:
                all_locations.add(pos)
        all_locations = list(all_locations)

        # 1. Try to pick up each object (execute_pick)
        for obj in object_names:
            try:
                obj_pos = positions[obj]
                # Find a location name that matches the object's position
                obj_loc = None
                for loc_name, loc_pos in positions.items():
                    if isinstance(loc_pos, tuple) and loc_pos == obj_pos:
                        obj_loc = loc_name
                        break
                if obj_loc is None and location_names:
                    obj_loc = location_names[0]
                print(f"[Exploration] Attempting to pick {obj} at {obj_loc}")
                obs, reward, done = execute_pick(env, task, obj, obj_loc)
                if done:
                    print(f"[Exploration] Successfully picked {obj}")
            except Exception as e:
                print(f"[Exploration] Could not pick {obj}: {e}")

        # 2. Try to pull each handle (execute_pull) to check for lock-known predicate
        for handle in handle_names:
            try:
                handle_pos = positions[handle]
                # Find associated drawer
                associated_drawer = None
                for drawer in drawer_names:
                    if handle in drawer or drawer in handle:
                        associated_drawer = drawer
                        break
                # Find a location name that matches the handle's position
                handle_loc = None
                for loc_name, loc_pos in positions.items():
                    if isinstance(loc_pos, tuple) and loc_pos == handle_pos:
                        handle_loc = loc_name
                        break
                if handle_loc is None and location_names:
                    handle_loc = location_names[0]
                print(f"[Exploration] Attempting to pull {associated_drawer} via {handle} at {handle_loc}")
                obs, reward, done = execute_pull(env, task, associated_drawer, handle, handle_loc)
                if done:
                    print(f"[Exploration] Successfully pulled {associated_drawer} via {handle}")
            except Exception as e:
                print(f"[Exploration] Could not pull {handle}: {e}")

        # 3. Try to open and close each drawer (execute_pull, execute_push)
        for drawer in drawer_names:
            try:
                # Find a handle for this drawer
                handle = None
                for h in handle_names:
                    if drawer in h or h in drawer:
                        handle = h
                        break
                if handle is None:
                    continue
                drawer_pos = positions[drawer]
                # Find a location name that matches the drawer's position
                drawer_loc = None
                for loc_name, loc_pos in positions.items():
                    if isinstance(loc_pos, tuple) and loc_pos == drawer_pos:
                        drawer_loc = loc_name
                        break
                if drawer_loc is None and location_names:
                    drawer_loc = location_names[0]
                print(f"[Exploration] Attempting to pull (open) {drawer} with {handle} at {drawer_loc}")
                obs, reward, done = execute_pull(env, task, drawer, handle, drawer_loc)
                if done:
                    print(f"[Exploration] Successfully opened {drawer}")
                print(f"[Exploration] Attempting to push (close) {drawer} at {drawer_loc}")
                obs, reward, done = execute_push(env, task, drawer, drawer_loc)
                if done:
                    print(f"[Exploration] Successfully closed {drawer}")
            except Exception as e:
                print(f"[Exploration] Could not open/close {drawer}: {e}")

        # 4. Try to move between locations (execute_go)
        if len(location_names) >= 2:
            from_loc = location_names[0]
            to_loc = location_names[1]
            try:
                print(f"[Exploration] Attempting to move from {from_loc} to {to_loc}")
                obs, reward, done = execute_go(env, task, from_loc, to_loc)
                if done:
                    print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}")
            except Exception as e:
                print(f"[Exploration] Could not move from {from_loc} to {to_loc}: {e}")

        # 5. Try to place each object into each drawer (execute_place)
        for obj in object_names:
            for drawer in drawer_names:
                try:
                    # Find a location for the drawer
                    drawer_pos = positions[drawer]
                    drawer_loc = None
                    for loc_name, loc_pos in positions.items():
                        if isinstance(loc_pos, tuple) and loc_pos == drawer_pos:
                            drawer_loc = loc_name
                            break
                    if drawer_loc is None and location_names:
                        drawer_loc = location_names[0]
                    print(f"[Exploration] Attempting to place {obj} into {drawer} at {drawer_loc}")
                    obs, reward, done = execute_place(env, task, obj, drawer, drawer_loc)
                    if done:
                        print(f"[Exploration] Successfully placed {obj} into {drawer}")
                except Exception as e:
                    print(f"[Exploration] Could not place {obj} into {drawer}: {e}")

        # 6. Try to sweep each object (execute_sweep)
        for obj in object_names:
            try:
                obj_pos = positions[obj]
                obj_loc = None
                for loc_name, loc_pos in positions.items():
                    if isinstance(loc_pos, tuple) and loc_pos == obj_pos:
                        obj_loc = loc_name
                        break
                if obj_loc is None and location_names:
                    obj_loc = location_names[0]
                print(f"[Exploration] Attempting to sweep {obj} at {obj_loc}")
                obs, reward, done = execute_sweep(env, task, obj, obj_loc)
                if done:
                    print(f"[Exploration] Successfully swept {obj}")
            except Exception as e:
                print(f"[Exploration] Could not sweep {obj}: {e}")

        # 7. Try to use gripper (execute_gripper)
        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(env, task)
            if done:
                print(f"[Exploration] Successfully used gripper")
        except Exception as e:
            print(f"[Exploration] Could not use gripper: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

        # === End of Exploration Phase ===

        # TODO: After exploration, analyze the logs and environment state to determine which predicate is missing.
        # This can be done by reviewing which actions failed and why, then updating the domain/problem accordingly.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()