# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (drawer-unlocked drawer1) is missing or needs to be explored.
        # We will use available skills to explore the lock state of drawer1.

        # Assumptions:
        # - There is a drawer named 'drawer1' and possibly a handle object for it.
        # - The robot can move to the drawer's location and interact with it.
        # - The available skills are: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Step 1: Find the location of drawer1 and its handle (if any)
        drawer_name = 'drawer1'
        handle_name = None
        drawer_location = None

        # Try to find handle and location from positions
        for obj_name, pos in positions.items():
            if 'handle' in obj_name and drawer_name in obj_name:
                handle_name = obj_name
            if drawer_name in obj_name and 'handle' not in obj_name:
                drawer_location = pos

        # Fallback: If not found, try default names
        if handle_name is None:
            for obj_name in positions:
                if 'handle' in obj_name:
                    handle_name = obj_name
                    break

        if drawer_location is None:
            if drawer_name in positions:
                drawer_location = positions[drawer_name]
            else:
                # Fallback: pick any location
                for obj_name, pos in positions.items():
                    if 'drawer' in obj_name:
                        drawer_location = pos
                        break

        # Step 2: Move robot to the drawer's location
        # Find robot's current location
        robot_location = None
        for obj_name, pos in positions.items():
            if 'robot' in obj_name:
                robot_location = pos
                break

        # If robot location is not found, assume a default
        if robot_location is None:
            robot_location = (0, 0, 0)

        # Use execute_go to move robot to drawer location
        try:
            print(f"[Exploration] Moving robot from {robot_location} to {drawer_location}")
            obs, reward, done = execute_go(env, task, from_location=robot_location, to_location=drawer_location)
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")

        # Step 3: Try to pick the handle (if any)
        picked_handle = False
        if handle_name is not None:
            try:
                print(f"[Exploration] Attempting to pick handle: {handle_name}")
                obs, reward, done = execute_pick(env, task, object_name=handle_name, location=drawer_location)
                picked_handle = True
            except Exception as e:
                print(f"[Exploration] Error during execute_pick on handle: {e}")

        # Step 4: Try to pull the drawer to check lock state (this may reveal lock-known or drawer-unlocked)
        if picked_handle:
            try:
                print(f"[Exploration] Attempting to pull drawer: {drawer_name} using handle: {handle_name}")
                obs, reward, done = execute_pull(env, task, drawer_name=drawer_name, handle_name=handle_name, location=drawer_location)
            except Exception as e:
                print(f"[Exploration] Error during execute_pull: {e}")
        else:
            print("[Exploration] Could not pick handle; skipping pull.")

        # Step 5: Check if the predicate (drawer-unlocked drawer1) is now true in the environment
        # This would typically be checked via the environment's state or observation
        # For demonstration, print the current observation
        try:
            current_obs = task.get_observation()
            print(f"[Exploration] Current observation after exploration: {current_obs}")
        except Exception as e:
            print(f"[Exploration] Error retrieving observation: {e}")

        # === End of Exploration Phase ===

        # (Optional) Continue with the main oracle plan if needed
        # For this task, the focus is on exploration to discover the missing predicate

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
