# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- EXPLORATION PHASE: Identify Missing Predicate (drawer-closed drawer1) ---
        # The feedback indicates that (drawer-closed drawer1) is a missing predicate.
        # We need to explore the environment to determine the state of drawer1.

        # For this, we will use available skills to check the state of the drawer.
        # Since we cannot define new skills, we use the available ones to interact with the drawer.

        # Let's assume the following object names and locations (replace with actual names if different):
        # - drawer1: the drawer to be checked
        # - handle1: the handle of drawer1
        # - robot initial location: 'loc_robot'
        # - drawer1 location: 'loc_drawer'
        # - handle1 location: 'loc_handle'
        # These should be present in the positions dictionary.

        # Try to extract relevant object and location names
        try:
            drawer_name = 'drawer1'
            handle_name = 'handle1'
            # Try to get locations from positions
            loc_robot = positions.get('robot', None)
            loc_drawer = positions.get(drawer_name, None)
            loc_handle = positions.get(handle_name, None)
        except Exception as e:
            print("[Exploration] Error extracting object positions:", e)
            loc_robot = None
            loc_drawer = None
            loc_handle = None

        # If locations are not found, skip exploration
        if loc_drawer is None or loc_handle is None:
            print("[Exploration] Drawer or handle location not found. Skipping exploration phase.")
        else:
            # Move robot to drawer location if not already there
            try:
                # If robot is not at drawer location, move there
                # Use execute_go (from, to)
                # For demonstration, we assume location names are available
                # If not, skip this step
                if loc_robot is not None and loc_robot != loc_drawer:
                    print(f"[Exploration] Moving robot from {loc_robot} to {loc_drawer}")
                    obs, reward, done = execute_go(env, task, loc_robot, loc_drawer)
                    loc_robot = loc_drawer
            except Exception as e:
                print("[Exploration] Error during execute_go:", e)

            # Try to pick the handle to check if the drawer is closed/locked
            try:
                print(f"[Exploration] Attempting to pick handle {handle_name} at {loc_handle}")
                obs, reward, done = execute_pick(env, task, handle_name, loc_handle)
            except Exception as e:
                print("[Exploration] Error during execute_pick (handle):", e)

            # Try to pull the drawer open (if possible)
            try:
                print(f"[Exploration] Attempting to pull drawer {drawer_name} using handle {handle_name} at {loc_drawer}")
                obs, reward, done = execute_pull(env, task, drawer_name, handle_name, loc_drawer)
            except Exception as e:
                print("[Exploration] Error during execute_pull (drawer):", e)

            # After attempting to pull, we can infer if the drawer was closed or locked
            # (In a real system, we would check the state, but here we just perform the actions)

        # --- END EXPLORATION PHASE ---

        # === MAIN TASK PLAN ===
        # After exploration, proceed with the main plan using available skills.
        # For demonstration, we show a generic sequence (replace with actual plan as needed):

        # Example: Pick an object from the floor and place it in the drawer

        # Assume object to pick is 'obj1' at location 'loc_obj1'
        obj_to_pick = 'obj1'
        loc_obj1 = positions.get(obj_to_pick, None)

        if loc_obj1 is not None and loc_drawer is not None:
            try:
                # Move to object location if needed
                if loc_robot is not None and loc_robot != loc_obj1:
                    print(f"[Task] Moving robot from {loc_robot} to {loc_obj1}")
                    obs, reward, done = execute_go(env, task, loc_robot, loc_obj1)
                    loc_robot = loc_obj1

                # Pick the object
                print(f"[Task] Picking object {obj_to_pick} at {loc_obj1}")
                obs, reward, done = execute_pick(env, task, obj_to_pick, loc_obj1)

                # Move to drawer location
                if loc_robot != loc_drawer:
                    print(f"[Task] Moving robot from {loc_robot} to {loc_drawer}")
                    obs, reward, done = execute_go(env, task, loc_robot, loc_drawer)
                    loc_robot = loc_drawer

                # Place the object in the drawer
                print(f"[Task] Placing object {obj_to_pick} in drawer {drawer_name} at {loc_drawer}")
                obs, reward, done = execute_place(env, task, obj_to_pick, drawer_name, loc_drawer)

            except Exception as e:
                print("[Task] Error during main plan execution:", e)
        else:
            print("[Task] Object or drawer location not found. Skipping main plan.")

        # You can add more steps as needed, using only the available skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()