# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We use the available skills to explore the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects/drawers and observe the environment's response.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to interact with all objects and drawers to see if any action fails due to a missing predicate
        # For demonstration, we attempt to pick and pull all objects/handles, and open/close drawers

        # Get all objects and drawers from positions (assuming keys are object/drawer names)
        object_names = [name for name in positions.keys() if 'object' in name]
        drawer_names = [name for name in positions.keys() if 'drawer' in name or 'Drawer' in name]
        handle_names = [name for name in positions.keys() if 'handle' in name or 'Handle' in name]
        location_names = [name for name in positions.keys() if 'location' in name or 'Location' in name or 'room' in name or 'Room' in name]

        # Fallback: If no explicit location names, use all unique positions as locations
        if not location_names:
            location_names = list(set([positions[name] for name in positions.keys()]))

        # For this exploration, we will try to:
        # 1. Move to each location
        # 2. Try to pick each object
        # 3. Try to pull each handle (to open drawers)
        # 4. Try to push each drawer (to close)
        # 5. Try to place objects in drawers

        # Keep track of which predicates might be missing based on action failures
        missing_predicates = set()

        # Helper: Try an action and catch exceptions, logging possible missing predicates
        def try_skill(skill_fn, *args, **kwargs):
            try:
                obs, reward, done = skill_fn(env, task, *args, **kwargs)
                return obs, reward, done
            except Exception as e:
                print(f"[Exploration] Exception during {skill_fn.__name__} with args {args}: {e}")
                # Heuristically guess missing predicate from error message
                msg = str(e)
                if 'locked' in msg or 'unlock' in msg:
                    missing_predicates.add('drawer-locked or drawer-unlocked')
                if 'open' in msg and 'drawer' in msg:
                    missing_predicates.add('drawer-open')
                if 'closed' in msg and 'drawer' in msg:
                    missing_predicates.add('drawer-closed')
                if 'full' in msg:
                    missing_predicates.add('drawer-full')
                if 'empty' in msg:
                    missing_predicates.add('drawer-empty')
                if 'partially' in msg:
                    missing_predicates.add('drawer-partially-filled')
                if 'hand' in msg and 'empty' in msg:
                    missing_predicates.add('hand-empty')
                if 'robot-free' in msg:
                    missing_predicates.add('robot-free')
                if 'on-floor' in msg:
                    missing_predicates.add('on-floor')
                if 'holding' in msg:
                    missing_predicates.add('holding')
                if 'handle-of' in msg:
                    missing_predicates.add('handle-of')
                # Add more heuristics as needed
                return None, None, None

        # 1. Move to each location (using execute_go)
        # For simplicity, assume robot starts at the first location
        if location_names:
            current_location = location_names[0]
            for loc in location_names:
                if loc == current_location:
                    continue
                print(f"[Exploration] Moving from {current_location} to {loc}")
                try_skill(execute_go, current_location, loc)
                current_location = loc

        # 2. Try to pick each object at each location
        for obj in object_names:
            for loc in location_names:
                print(f"[Exploration] Trying to pick {obj} at {loc}")
                try_skill(execute_pick, obj, loc)

        # 3. Try to pull each handle at each location (simulate opening drawers)
        for drawer in drawer_names:
            for handle in handle_names:
                for loc in location_names:
                    print(f"[Exploration] Trying to pull {drawer} with handle {handle} at {loc}")
                    try_skill(execute_pull, drawer, handle, loc)

        # 4. Try to push each drawer at each location (simulate closing drawers)
        for drawer in drawer_names:
            for loc in location_names:
                print(f"[Exploration] Trying to push {drawer} at {loc}")
                try_skill(execute_push, drawer, loc)

        # 5. Try to place each object in each drawer at each location
        for obj in object_names:
            for drawer in drawer_names:
                for loc in location_names:
                    print(f"[Exploration] Trying to place {obj} in {drawer} at {loc}")
                    try_skill(execute_place, obj, drawer, loc)

        # 6. Try other skills if needed (sweep, rotate, gripper)
        for obj in object_names:
            for loc in location_names:
                print(f"[Exploration] Trying to sweep {obj} at {loc}")
                try_skill(execute_sweep, obj, loc)

        try:
            print(f"[Exploration] Trying to use gripper")
            try_skill(execute_gripper)
        except Exception as e:
            print(f"[Exploration] Exception during execute_gripper: {e}")

        # === Report Missing Predicates ===
        if missing_predicates:
            print("=== Exploration Result: Possible Missing Predicates ===")
            for pred in missing_predicates:
                print(f"  - {pred}")
        else:
            print("=== Exploration Result: No obvious missing predicates detected ===")

        # === End of Exploration Phase ===
        # At this point, you can use the exploration results to update your domain/problem files or inform the planner.

        # === (Optional) Main Task Plan Execution ===
        # If you have an oracle plan, you would execute it here step by step using the available skills.
        # For this task, the focus is on the exploration phase to identify missing predicates.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
