# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Define object and location names based on observation and feedback ===
        # These should match the names in the PDDL and environment
        # Example objects: dice1, dice2, handle1, handle2, handle3, drawer1, drawer2, drawer3
        # Example locations: floor, drawer-area, ready-pose

        # For this task, we want to put a target object (e.g., dice1) into drawer1, which is open and empty
        # We'll use dice1 as the target object, drawer1 as the target drawer, and ready-pose as the starting location

        target_object = "dice1"
        target_drawer = "drawer1"
        handle_object = "handle1"  # handle for drawer1
        start_location = "ready-pose"
        floor_location = "floor"
        drawer_area_location = "drawer-area"

        # === Exploration Phase: Check for missing predicates or knowledge ===
        # According to the feedback, we need to ensure the object is on the floor and not already in the drawer
        # If the object is not found, or already in the drawer, abort

        # For demonstration, we assume get_object_positions() returns a dict like:
        # {'dice1': {'location': 'floor'}, ...}
        try:
            obj_info = positions.get(target_object, {})
            obj_location = obj_info.get('location', None)
        except Exception as e:
            print(f"[Error] Could not retrieve position for {target_object}: {e}")
            return

        if obj_location != floor_location:
            print(f"[Exploration] {target_object} is not on the floor (found at {obj_location}). Aborting.")
            return

        # === Plan Execution ===
        # 1. Move robot to the floor location if not already there
        # 2. Pick up the target object from the floor
        # 3. Move to the drawer area (if needed)
        # 4. Place the object into the open drawer

        # Step 1: Move to floor location if not already there
        try:
            robot_pos = positions.get('robot', {}).get('location', start_location)
        except Exception as e:
            print(f"[Error] Could not retrieve robot position: {e}")
            robot_pos = start_location

        if robot_pos != floor_location:
            print(f"[Action] Moving robot from {robot_pos} to {floor_location}")
            try:
                obs, reward, done, info = execute_go(env, task, robot_pos, floor_location)
            except Exception as e:
                print(f"[Error] execute_go failed: {e}")
                return
        else:
            print(f"[Info] Robot already at {floor_location}")

        # Step 2: Pick up the target object
        print(f"[Action] Picking up {target_object} from {floor_location}")
        try:
            obs, reward, done, info = execute_pick(env, task, target_object, floor_location)
        except Exception as e:
            print(f"[Error] execute_pick failed: {e}")
            return

        # Step 3: Move to drawer area if not already there
        # (Assume the drawer is at drawer-area)
        try:
            robot_pos = positions.get('robot', {}).get('location', floor_location)
        except Exception as e:
            robot_pos = floor_location

        if robot_pos != drawer_area_location:
            print(f"[Action] Moving robot from {robot_pos} to {drawer_area_location}")
            try:
                obs, reward, done, info = execute_go(env, task, robot_pos, drawer_area_location)
            except Exception as e:
                print(f"[Error] execute_go failed: {e}")
                return
        else:
            print(f"[Info] Robot already at {drawer_area_location}")

        # Step 4: Place the object into the open drawer
        print(f"[Action] Placing {target_object} into {target_drawer} at {drawer_area_location}")
        try:
            obs, reward, done, info = execute_place(env, task, target_object, target_drawer, drawer_area_location)
        except Exception as e:
            print(f"[Error] execute_place failed: {e}")
            return

        print("[Task] Successfully placed the object into the drawer.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()