# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate (robot-free) ===
        # Feedback indicates (robot-free) is missing or needs to be checked.
        # We will use available skills to explore the environment and check for this predicate.

        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We will attempt to use the skills in a safe exploratory sequence to check for the 'robot-free' state.
        # Since we do not have the exact object and location names, we will use the positions dictionary to iterate.

        # 1. Try to move the robot to a different location using execute_go
        location_names = list(positions.keys())
        if len(location_names) >= 2:
            from_loc = location_names[0]
            to_loc = location_names[1]
            try:
                print(f"[Exploration] Moving robot from {from_loc} to {to_loc} using execute_go.")
                obs, reward, done = execute_go(env, task, from_loc, to_loc)
                print("[Exploration] execute_go completed.")
            except Exception as e:
                print(f"[Exploration] execute_go failed: {e}")

        # 2. Try to pick up an object using execute_pick
        # Find an object on the floor and its location
        object_on_floor = None
        object_location = None
        for obj_name, pos in positions.items():
            if 'object' in obj_name or 'obj' in obj_name:
                object_on_floor = obj_name
                object_location = pos
                break
        if object_on_floor is not None and object_location is not None:
            try:
                print(f"[Exploration] Attempting to pick up {object_on_floor} at {object_location} using execute_pick.")
                obs, reward, done = execute_pick(env, task, object_on_floor, object_location)
                print("[Exploration] execute_pick completed.")
            except Exception as e:
                print(f"[Exploration] execute_pick failed: {e}")

        # 3. Try to place the object in a drawer if possible using execute_place
        # Find a drawer and its location
        drawer_name = None
        drawer_location = None
        for obj_name, pos in positions.items():
            if 'drawer' in obj_name:
                drawer_name = obj_name
                drawer_location = pos
                break
        if object_on_floor is not None and drawer_name is not None and drawer_location is not None:
            try:
                print(f"[Exploration] Attempting to place {object_on_floor} in {drawer_name} at {drawer_location} using execute_place.")
                obs, reward, done = execute_place(env, task, object_on_floor, drawer_name, drawer_location)
                print("[Exploration] execute_place completed.")
            except Exception as e:
                print(f"[Exploration] execute_place failed: {e}")

        # 4. Try to open the drawer using execute_pull if possible
        # Find a handle object for the drawer
        handle_name = None
        for obj_name in positions.keys():
            if 'handle' in obj_name:
                handle_name = obj_name
                break
        if drawer_name is not None and handle_name is not None and drawer_location is not None:
            try:
                print(f"[Exploration] Attempting to pull {drawer_name} using handle {handle_name} at {drawer_location} with execute_pull.")
                obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_location)
                print("[Exploration] execute_pull completed.")
            except Exception as e:
                print(f"[Exploration] execute_pull failed: {e}")

        # 5. Try to push (close) the drawer using execute_push
        if drawer_name is not None and drawer_location is not None:
            try:
                print(f"[Exploration] Attempting to push (close) {drawer_name} at {drawer_location} using execute_push.")
                obs, reward, done = execute_push(env, task, drawer_name, drawer_location)
                print("[Exploration] execute_push completed.")
            except Exception as e:
                print(f"[Exploration] execute_push failed: {e}")

        # 6. Try to sweep an object using execute_sweep
        if object_on_floor is not None and object_location is not None:
            try:
                print(f"[Exploration] Attempting to sweep {object_on_floor} at {object_location} using execute_sweep.")
                obs, reward, done = execute_sweep(env, task, object_on_floor, object_location)
                print("[Exploration] execute_sweep completed.")
            except Exception as e:
                print(f"[Exploration] execute_sweep failed: {e}")

        # 7. Try to use the gripper using execute_gripper
        try:
            print("[Exploration] Attempting to use gripper with execute_gripper.")
            obs, reward, done = execute_gripper(env, task)
            print("[Exploration] execute_gripper completed.")
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        # After each action, check if the 'robot-free' predicate is now true in the environment.
        # This would typically be done by querying the environment state or observation.
        # For demonstration, we print a message after each action.

        print("[Exploration] Exploration phase complete. Check logs or environment state for 'robot-free' predicate.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()