# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (robot-free) ===
        # The feedback indicates that the predicate (robot-free) is missing or not being set as expected.
        # We will perform a sequence of actions using only the available skills to explore the state transitions
        # and check if/when (robot-free) becomes true.

        # Available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # For exploration, we will:
        # 1. Move the robot to a location with an object.
        # 2. Attempt to pick the object.
        # 3. Place the object in a drawer (if possible).
        # 4. Open/close the drawer.
        # 5. After each action, check if the robot is "free" (i.e., robot-free predicate is true in the environment state).

        # For this, we need to know:
        # - The names of at least one object, one drawer, one handle, and one location.
        # - The mapping of objects to drawers/handles/locations.
        # We'll use the positions dictionary and assume it contains keys like 'object_1', 'drawer_1', 'handle_1', 'location_1', etc.

        # --- Example object/drawer/handle/location selection ---
        # (You may need to adjust these keys based on your environment's actual object names.)
        try:
            # Try to extract at least one object, drawer, handle, and location from positions
            object_names = [k for k in positions.keys() if 'object' in k or 'obj' in k]
            drawer_names = [k for k in positions.keys() if 'drawer' in k]
            handle_names = [k for k in positions.keys() if 'handle' in k]
            location_names = [k for k in positions.keys() if 'location' in k or 'room' in k or 'pos' in k]

            if not object_names or not drawer_names or not location_names:
                print("[Exploration] Could not find required objects/drawers/locations in positions. Exploration aborted.")
                return

            obj = object_names[0]
            drawer = drawer_names[0]
            location = location_names[0]
            # If handle is needed for pull, try to get it
            handle = handle_names[0] if handle_names else None

            print(f"[Exploration] Using object: {obj}, drawer: {drawer}, location: {location}, handle: {handle}")

            # 1. Move to the location
            print(f"[Exploration] Moving robot to {location}")
            obs, reward, done = execute_go(env, task, from_location=None, to_location=location)
            # from_location can be None if not tracked; skill should handle current position

            # 2. Try to pick the object
            print(f"[Exploration] Attempting to pick {obj} at {location}")
            obs, reward, done = execute_pick(env, task, object_name=obj, location=location)

            # 3. Try to open the drawer (if handle is available)
            if handle:
                print(f"[Exploration] Attempting to pull (open) {drawer} using handle {handle} at {location}")
                obs, reward, done = execute_pull(env, task, drawer_name=drawer, handle_name=handle, location=location)

            # 4. Try to place the object in the drawer
            print(f"[Exploration] Attempting to place {obj} in {drawer} at {location}")
            obs, reward, done = execute_place(env, task, object_name=obj, drawer_name=drawer, location=location)

            # 5. Try to close the drawer
            print(f"[Exploration] Attempting to push (close) {drawer} at {location}")
            obs, reward, done = execute_push(env, task, drawer_name=drawer, location=location)

            # 6. After each action, check for 'robot-free' predicate in the environment state
            # (Assume task.get_state() or similar returns a dict of predicates)
            if hasattr(task, 'get_state'):
                state = task.get_state()
                if 'robot-free' in state and state['robot-free']:
                    print("[Exploration] Predicate (robot-free) is TRUE after actions.")
                else:
                    print("[Exploration] Predicate (robot-free) is FALSE or missing after actions.")
            else:
                print("[Exploration] Cannot directly check for (robot-free) predicate in task state.")

        except Exception as e:
            print(f"[Exploration] Exception during exploration: {e}")

        # === End of Exploration Phase ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()