# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available skills to explore the environment and infer which predicate is missing.
        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We will attempt to use each skill in a safe, exploratory way and observe the environment's response.
        # This will help us deduce which predicate is missing or required for successful planning.

        # For demonstration, we will:
        # 1. Try to move the robot to each location (execute_go)
        # 2. Try to pick up each object (execute_pick)
        # 3. Try to open and close drawers (execute_pull, execute_push)
        # 4. Try to place objects (execute_place)
        # 5. Try to sweep and rotate (execute_sweep, execute_rotate)
        # 6. Try to use the gripper (execute_gripper)
        # We will catch and print any exceptions to help identify missing predicates.

        # Get all objects and locations from positions
        object_names = [name for name in positions if positions[name].get('type', '') == 'object']
        drawer_names = [name for name in positions if positions[name].get('type', '') == 'drawer']
        location_names = [name for name in positions if positions[name].get('type', '') == 'location']
        handle_names = [name for name in positions if positions[name].get('type', '') == 'handle']

        # Fallback: If type info is not available, use all keys as objects/locations
        if not object_names:
            object_names = [name for name in positions if 'object' in name]
        if not drawer_names:
            drawer_names = [name for name in positions if 'drawer' in name]
        if not location_names:
            location_names = [name for name in positions if 'room' in name or 'location' in name or 'loc' in name]
        if not handle_names:
            handle_names = [name for name in positions if 'handle' in name]

        # If still empty, just use all keys
        if not object_names:
            object_names = list(positions.keys())
        if not location_names:
            location_names = list(positions.keys())

        # Try to get the robot's current location
        robot_location = None
        for loc in location_names:
            if positions[loc].get('robot_at', False):
                robot_location = loc
                break
        if robot_location is None and location_names:
            robot_location = location_names[0]

        print("[Exploration] Robot initial location:", robot_location)
        print("[Exploration] Objects:", object_names)
        print("[Exploration] Drawers:", drawer_names)
        print("[Exploration] Handles:", handle_names)
        print("[Exploration] Locations:", location_names)

        # 1. Try to move to each location
        for to_loc in location_names:
            if to_loc == robot_location:
                continue
            try:
                print(f"[Exploration] Trying to move from {robot_location} to {to_loc} using execute_go")
                obs, reward, done = execute_go(env, task, from_location=robot_location, to_location=to_loc)
                robot_location = to_loc
                if done:
                    print("[Exploration] Task ended after move!")
                    return
            except Exception as e:
                print(f"[Exploration] execute_go failed from {robot_location} to {to_loc}: {e}")

        # 2. Try to pick up each object
        for obj in object_names:
            try:
                print(f"[Exploration] Trying to pick up {obj} at {robot_location} using execute_pick")
                obs, reward, done = execute_pick(env, task, object_name=obj, location=robot_location)
                if done:
                    print("[Exploration] Task ended after pick!")
                    return
            except Exception as e:
                print(f"[Exploration] execute_pick failed for {obj} at {robot_location}: {e}")

        # 3. Try to open each drawer using each handle
        for drawer in drawer_names:
            for handle in handle_names:
                try:
                    print(f"[Exploration] Trying to pull {drawer} with handle {handle} at {robot_location} using execute_pull")
                    obs, reward, done = execute_pull(env, task, drawer_name=drawer, handle_name=handle, location=robot_location)
                    if done:
                        print("[Exploration] Task ended after pull!")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_pull failed for {drawer} with {handle} at {robot_location}: {e}")

        # 4. Try to close each drawer
        for drawer in drawer_names:
            try:
                print(f"[Exploration] Trying to push (close) {drawer} at {robot_location} using execute_push")
                obs, reward, done = execute_push(env, task, drawer_name=drawer, location=robot_location)
                if done:
                    print("[Exploration] Task ended after push!")
                    return
            except Exception as e:
                print(f"[Exploration] execute_push failed for {drawer} at {robot_location}: {e}")

        # 5. Try to place each object in each drawer
        for obj in object_names:
            for drawer in drawer_names:
                try:
                    print(f"[Exploration] Trying to place {obj} in {drawer} at {robot_location} using execute_place")
                    obs, reward, done = execute_place(env, task, object_name=obj, drawer_name=drawer, location=robot_location)
                    if done:
                        print("[Exploration] Task ended after place!")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_place failed for {obj} in {drawer} at {robot_location}: {e}")

        # 6. Try to sweep each object
        for obj in object_names:
            try:
                print(f"[Exploration] Trying to sweep {obj} at {robot_location} using execute_sweep")
                obs, reward, done = execute_sweep(env, task, object_name=obj, location=robot_location)
                if done:
                    print("[Exploration] Task ended after sweep!")
                    return
            except Exception as e:
                print(f"[Exploration] execute_sweep failed for {obj} at {robot_location}: {e}")

        # 7. Try to rotate each object (if applicable)
        for obj in object_names:
            try:
                print(f"[Exploration] Trying to rotate {obj} at {robot_location} using execute_rotate")
                obs, reward, done = execute_rotate(env, task, object_name=obj, location=robot_location)
                if done:
                    print("[Exploration] Task ended after rotate!")
                    return
            except Exception as e:
                print(f"[Exploration] execute_rotate failed for {obj} at {robot_location}: {e}")

        # 8. Try to use the gripper
        try:
            print(f"[Exploration] Trying to use gripper using execute_gripper")
            obs, reward, done = execute_gripper(env, task)
            if done:
                print("[Exploration] Task ended after gripper!")
                return
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Exploration phase completed. Check above for failed actions and missing predicates.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()